<?php

namespace App\Helpers;

class QuestionHelper
{
    /**
     * ตรวจจับเลขข้อจากข้อความคำถาม
     * รองรับรูปแบบ: 1.2.3, 1.2.3.4, 1.2.3.4.5, etc.
     */
    public static function extractQuestionNumber($questionText)
    {
        // รูปแบบ regex สำหรับตรวจจับเลขข้อ
        // รองรับ: 1.2.3, 1.2.3.4, 1.2.3.4.5, etc.
        $pattern = '/^(\d+(?:\.\d+)*)\s*(.+)$/';
        
        if (preg_match($pattern, trim($questionText), $matches)) { //preg_match คือ ฟังก์ชันที่ใช้ตรวจสอบว่าข้อความตรงกับรูปแบบ regex(Regular Expression) หรือไม่
            return [
                'number' => $matches[1], // เลขข้อ เช่น "4.2.1.1"
                'text' => trim($matches[2]) // ข้อความ เช่น "วิธีการติดตั้ง หมายรวมถึง"
            ];
        }
        
        return [
            'number' => null,
            'text' => $questionText
        ];
    }

    /**
     * ตรวจสอบว่าเป็นเลขข้อที่ถูกต้องหรือไม่
     */
    public static function isValidQuestionNumber($number)
    {
        if (!$number) return false;
        
        // ตรวจสอบรูปแบบเลขข้อ (ต้องเป็นตัวเลขที่คั่นด้วยจุด)
        return preg_match('/^\d+(?:\.\d+)*$/', $number);
    }

    /**
     * จัดรูปแบบเลขข้อให้สวยงาม
     */
    public static function formatQuestionNumber($number)
    {
        if (!$number) return '';
        
        // แบ่งส่วนของเลขข้อ
        $parts = explode('.', $number);
        
        // จัดรูปแบบให้เป็น "ข้อ 4.2.1.1"
        return 'ข้อ ' . $number;
    }

    /**
     * สร้างเลขข้อถัดไป
     */
    public static function generateNextQuestionNumber($parentNumber = null, $siblingCount = 0)
    {
        if ($parentNumber) {
            return $parentNumber . '.' . ($siblingCount + 1);
        }
        
        return ($siblingCount + 1) . '.1';
    }

    /**
     * ตรวจสอบว่าเลขข้อซ้ำหรือไม่
     */
    public static function isDuplicateQuestionNumber($number, $questionGroupId, $excludeQuestionId = null)
    {
        // ดึงคำถามทั้งหมดในกลุ่ม
        $questions = \App\Models\Question::where('question_group_id', $questionGroupId);
        
        if ($excludeQuestionId) {
            $questions->where('id', '!=', $excludeQuestionId);
        }
        
        $questions = $questions->get();
        
        // ตรวจสอบเลขข้อซ้ำโดยตรง
        foreach ($questions as $question) {
            $existingNumber = self::extractQuestionNumber($question->question_text)['number'];
            if ($existingNumber === $number) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * แยกข้อความคำถามเป็นเลขข้อและเนื้อหา
     */
    public static function parseQuestionText($questionText)
    {
        $result = self::extractQuestionNumber($questionText);
        
        return [
            'question_number' => $result['number'],
            'question_content' => $result['text'],
            'formatted_number' => $result['number'] ? self::formatQuestionNumber($result['number']) : '',
            'has_number' => !is_null($result['number'])
        ];
    }

    /**
     * สร้าง sort key สำหรับการเรียงลำดับเลขข้อแบบลำดับชั้น
     */
    public static function generateSortKey($questionNumber)
    {
        if (!$questionNumber) return '999999.999999.999999.999999.999999';
        
        $parts = explode('.', $questionNumber);
        $sortKey = '';
        
        // สร้าง sort key โดยเติม 0 ด้านหน้าให้แต่ละส่วนมี 6 หลัก
        foreach ($parts as $part) {
            $sortKey .= str_pad($part, 6, '0', STR_PAD_LEFT) . '.';
        }
        
        // ลบจุดท้ายสุด
        $sortKey = rtrim($sortKey, '.');
        
        // เติม .000000 สำหรับส่วนที่เหลือ (สูงสุด 5 ระดับ)
        // ใช้ 000000 แทน 999999 เพื่อให้เรียงลำดับถูกต้อง
        while (substr_count($sortKey, '.') < 4) {
            $sortKey .= '.000000';
        }
        
        return $sortKey;
    }

    /**
     * เรียงลำดับคำถามตามเลขข้อแบบลำดับชั้น
     */
    public static function sortQuestionsByNumber($questions)
    {
        return $questions->sortBy(function($question) {
            $number = self::extractQuestionNumber($question->question_text)['number'];
            return self::generateSortKey($number);
        });
    }

    /**
     * จัดรูปแบบเลขข้อให้แสดงเฉพาะตัวเลข (ไม่มีคำว่า "ข้อ")
     */
    public static function formatQuestionNumberOnly($number)
    {
        if (!$number) return '';
        return $number;
    }

/**
 * แปลงคำถามเป็น D3.js data structure
 */
public static function toD3Data($questions)
{
    $d3Data = [];
    
    foreach ($questions as $question) {
        $d3Data[] = self::convertQuestionToD3($question);
    }
    
    return $d3Data;
}

/**
 * แปลงคำถามเดี่ยวเป็น D3.js format
 */
public static function convertQuestionToD3($question)
{
    $data = [
        'id' => $question->id,
        'name' => $question->question_text,
        'description' => $question->description,
        'type' => $question->question_type,
        'logic_operator' => $question->logic_operator,
        'use_in_evaluation' => $question->use_in_evaluation,
        'question_number' => self::extractQuestionNumber($question->question_text)['number'],
        'sort_order' => $question->sort_order,
        'children' => []
    ];

    // เพิ่มคำถามย่อย (recursive)
    if ($question->hasChildren()) {
        foreach ($question->children as $child) {
            $data['children'][] = self::convertQuestionToD3($child);
        }
    }

    return $data;
}

/**
 * แปลงคำถามเป็น D3.js data structure สำหรับกลุ่มเดียว
 */
public static function questionGroupToD3($questionGroup)
{
    // ดึงคำถามแม่ (ไม่มี parent)
    $rootQuestions = $questionGroup->questions()
        ->whereNull('parent_id')
        ->with(['children' => function($query) {
            $query->orderBy('sort_order');
        }])
        ->orderBy('sort_order')
        ->get();

    return self::toD3Data($rootQuestions);
}

/**
 * แปลงคำถามทั้งหมดเป็น D3.js data structure
 */
public static function allQuestionsToD3()
{
    $groups = \App\Models\QuestionGroup::with(['questions' => function($query) {
        $query->whereNull('parent_id')
              ->with(['children' => function($subQuery) {
                  $subQuery->orderBy('sort_order');
              }])
              ->orderBy('sort_order');
    }])
    ->where('is_active', true)
    ->orderBy('name')
    ->get();

    $d3Data = [];
    foreach ($groups as $group) {
        $d3Data[] = [
            'id' => 'group_' . $group->id,
            'name' => $group->name,
            'type' => 'group',
            'children' => self::toD3Data($group->questions)
        ];
    }

    return $d3Data;
}

/**
 * สร้าง D3.js data structure สำหรับ tree layout
 */
public static function createTreeData($questions)
{
    // สร้าง root node ถ้ามีคำถามหลายตัว
    if (count($questions) > 1) {
        return [
            'id' => 'root',
            'name' => 'คำถามทั้งหมด',
            'type' => 'root',
            'children' => self::toD3Data($questions)
        ];
    }

    // ถ้ามีคำถามเดียว ให้ใช้เป็น root
    return self::convertQuestionToD3($questions->first());
}
} 