<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Assessment;
use App\Models\Answer;
use App\Models\Question;
use App\Models\QuestionGroup;
use App\Models\Customer;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Helpers\QuestionHelper; // เพิ่ม import

class AdminController extends Controller
{
    /**
     * แสดงหน้า dashboard สำหรับ admin
     */
    public function dashboard(Request $request)
    {
        // สถิติโดยรวม
        $stats = [
            'total_users' => Customer::count(),
            'total_products' => Product::count(),
            'total_assessments' => Assessment::count(),
            'admin_users' => Customer::where('level', 'admin')->count(),
            'regular_users' => Customer::where('level', 'client')->count(),
        ];

        // ดึงข้อมูลบริษัทสำหรับ dropdown
        $agencies = \App\Models\Agency::where('active', 1)->orderBy('Agency_Name')->get();

        // Query ผลิตภัณฑ์พร้อมการกรอง
        $query = Product::with(['user.agency']);

        // กรองตามประเภทผลิตภัณฑ์
        if ($request->filled('product_type')) {
            $query->where('product_type', $request->product_type);
        }

        // กรองตามบริษัท
        if ($request->filled('agency_id')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('Agency_ID', $request->agency_id);
            });
        }

        // กรองตามชื่อลูกค้า
        if ($request->filled('customer_name')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('Customer_Name', 'like', '%' . $request->customer_name . '%');
            });
        }

        // กรองตามชื่อผลิตภัณฑ์หรือรุ่น
        if ($request->filled('product_name')) {
            $query->where(function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->product_name . '%')
                  ->orWhere('model', 'like', '%' . $request->product_name . '%');
            });
        }

        // เรียงตามวันที่ล่าสุด
        $products = $query->orderBy('DtmIns', 'desc')->paginate(15);

        return view('admin.dashboard', compact('stats', 'products', 'agencies'));
    }

    /**
     * แสดงรายการผู้ใช้ทั้งหมด
     */
    public function users()
    {
        $users = Customer::with('agency')
            ->orderBy('DtmIns', 'desc')
            ->paginate(15);

        return view('admin.users.index', compact('users'));
    }

    /**
     * แสดงรายการผลิตภัณฑ์ทั้งหมด
     */
    public function products()
    {
        $products = Product::with(['user', 'assessments'])
            ->orderBy('DtmIns', 'desc')
            ->paginate(15);

        return view('admin.products.index', compact('products'));
    }

    /**
     * แสดงรายการการประเมินทั้งหมด
     */
    public function assessments()
    {
        $assessments = Assessment::with(['product', 'user'])
            ->orderBy('DtmIns', 'desc')
            ->paginate(15);

        return view('admin.assessments.index', compact('assessments'));
    }

    /**
     * แสดงรายงานสถิติ
     */
    public function reports()
    {
        // สถิติรายเดือน
        $monthly_stats = Assessment::select(
            DB::raw('YEAR(DtmIns) as year'),
            DB::raw('MONTH(DtmIns) as month'),
            DB::raw('count(*) as count')
        )
        ->groupBy('year', 'month')
        ->orderBy('year', 'desc')
        ->orderBy('month', 'desc')
        ->take(12)
        ->get();

        // สถิติตามบริษัท
        $agency_stats = Customer::select(
            'Agency_ID',
            DB::raw('count(*) as user_count')
        )
        ->whereNotNull('Agency_ID')
        ->with('agency')
        ->groupBy('Agency_ID')
        ->get();

        return view('admin.reports', compact('monthly_stats', 'agency_stats'));
    }

    /**
     * แสดงฟอร์มแก้ไขผลิตภัณฑ์
     */
    public function editProduct(Product $product)
    {
        return view('admin.products.edit', compact('product'));
    }

    /**
     * อัปเดตข้อมูลผลิตภัณฑ์
     */
    public function updateProduct(Request $request, Product $product)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'model' => 'nullable|string|max:255',
            'log_receive_method' => 'required|in:syslog_udp,syslog_tcp,syslog_both,other',
            'product_type' => 'required|in:hw_sw_log_server,sw_log_server',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        try {
            // จัดการอัพโหลดรูปภาพ
            if ($request->hasFile('image')) {
                // ลบรูปภาพเก่า
                if ($product->image && Storage::disk('public')->exists($product->image)) {
                    Storage::disk('public')->delete($product->image);
                }
                
                $path = $request->file('image')->store('products_images', 'public');
                $validated['image'] = $path;
            }

            $product->update($validated);

            return redirect()->route('admin.dashboard')
                ->with('success', 'อัปเดตข้อมูลผลิตภัณฑ์เรียบร้อยแล้ว');

        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'เกิดข้อผิดพลาดในการอัปเดตข้อมูลผลิตภัณฑ์');
        }
    }

    /**
     * ลบผลิตภัณฑ์
     */
    public function deleteProduct(Product $product)
    {
        try {
            // ลบรูปภาพ
            if ($product->image && Storage::disk('public')->exists($product->image)) {
                Storage::disk('public')->delete($product->image);
            }

            // ลบการประเมินที่เกี่ยวข้อง
            $product->assessments()->delete();
            
            // ลบผลิตภัณฑ์
            $product->delete();

            return response()->json([
                'success' => true,
                'message' => 'ลบผลิตภัณฑ์เรียบร้อยแล้ว'
            ]);

        } catch (\Exception $e) {
            Log::error('Error deleting product', [
                'product_id' => $product->id,
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการลบผลิตภัณฑ์: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * ดูรายงานผลิตภัณฑ์ (สำหรับ admin)
     */
    public function viewReport(Product $product)
    {
        // ดึงข้อมูลการประเมิณของผลิตภัณฑ์ พร้อมคำตอบและคำถาม
        $assessment = $product->assessments()
            ->with(['answers.question'])
            ->latest()
            ->first();

        if(!$assessment){
            return view('products.report',[
                'product'=>$product,
                'assessment' => null,
                'answeredQuestions' => [],
            ]);
        }

        // ดึงเฉพาะคำถามที่มีคำตอบใน assessment
        $answeredQuestions = collect();
        
        foreach ($assessment->answers as $answer) {
            $question = $answer->question;
            if ($question) {
                // เพิ่มข้อมูลคำตอบเข้าไปในคำถาม
                $question->answer = $answer;
                $answeredQuestions->push($question);
            }
        }
        
        // เรียงลำดับตามเลขข้อที่ถูกต้อง
        $answeredQuestions = $answeredQuestions->sortBy(function($question) {
            // ใช้ question_number_sort_key ถ้ามี หรือ fallback ไปใช้ sort_order
            return $question->question_number_sort_key ?? $question->sort_order ?? 0;
        });

        return view('products.report', compact('product','assessment','answeredQuestions'));
    }

    /**
     * เปลี่ยนสถานะผู้ใช้
     */
    public function toggleUserStatus(Customer $user)
    {
        try {
            $oldStatus = $user->On_Off_Customer;
            $newStatus = $oldStatus == 'On' ? 'Off' : 'On';
            
            $user->update([
                'On_Off_Customer' => $newStatus,
                'CusUdp_ID' => Auth::user()->Customer_ID,
                'DtmUdp' => now()
            ]);

            // บันทึก log
            Log::info('User status changed', [
                'user_id' => $user->Customer_ID,
                'user_name' => $user->Customer_Name,
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'changed_by' => Auth::user()->Customer_ID
            ]);

            $message = $newStatus == 'Off' 
                ? "ปิดใช้งานบัญชี {$user->Customer_Name} เรียบร้อยแล้ว" 
                : "เปิดใช้งานบัญชี {$user->Customer_Name} เรียบร้อยแล้ว";

            return response()->json([
                'success' => true,
                'message' => $message,
                'new_status' => $newStatus
            ]);
        } catch (\Exception $e) {
            Log::error('Error changing user status', [
                'user_id' => $user->Customer_ID,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการเปลี่ยนสถานะ'
            ], 500);
        }
    }

    /**
     * ดูรายละเอียดการประเมิน
     */
    public function viewAssessment(Assessment $assessment)
    {
        $assessment->load(['product', 'user', 'answers.question']);
        
        return view('admin.assessments.view', compact('assessment'));
    }

    /**
     * ดูรายงานการประเมิน
     */
    public function assessmentReport(Assessment $assessment)
    {
        $assessment->load(['product', 'user', 'answers.question']);
        
        // ดึงคำตอบจาก assessment พร้อมเรียงลำดับตามเลขข้อ
        $answers = $assessment->answers->sortBy(function($answer) {
            $parsed = \App\Helpers\QuestionHelper::parseQuestionText($answer->question->question_text);
            return \App\Helpers\QuestionHelper::generateSortKey($parsed['question_number']);
        });

        return view('admin.assessments.report', compact('assessment', 'answers'));
    }

    /**
     * ลบการประเมิน
     */
    public function deleteAssessment(Assessment $assessment)
    {
        // ตรวจสอบสิทธิ์ - อนุญาตให้เจ้าของผลิตภัณฑ์, คนในบริษัทเดียวกัน, หรือ admin ลบได้
        $user = Auth::user();
        $hasDeleteAccess = false;
        
        // เจ้าของผลิตภัณฑ์
        if($user->Customer_ID === $assessment->product->user_id) {
            $hasDeleteAccess = true;
        }
        // Admin
        elseif($user->Level === 'admin') {
            $hasDeleteAccess = true;
        }
        // คนในบริษัทเดียวกัน
        elseif($user->Agency_ID && $assessment->product->user && $assessment->product->user->Agency_ID && $user->Agency_ID === $assessment->product->user->Agency_ID) {
            $hasDeleteAccess = true;
        }
        
        if(!$hasDeleteAccess) {
            return response()->json([
                'success' => false,
                'message' => 'คุณไม่มีสิทธิ์ในการลบการประเมินนี้'
            ], 403);
        }
        
        try {
            // ลบคำตอบที่เกี่ยวข้อง
            $assessment->answers()->delete();
            
            // ลบการประเมิน
            $assessment->delete();

            return response()->json([
                'success' => true,
                'message' => 'ลบการประเมินเรียบร้อยแล้ว'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการลบการประเมิน'
            ], 500);
        }
    }

    /**
     * แสดงรายการ Question Groups
     */
    public function questionGroups()
    {
        $groups = QuestionGroup::with(['questions' => function($query) {
            $query->whereNull('parent_id')->orderBy('sort_order');
        }, 'questions.children' => function($query) {
            $query->orderBy('sort_order');
        }])
        ->orderBy('DtmIns', 'desc')
        ->paginate(10);

        return view('admin.question-groups.index', compact('groups'));
    }

    /**
     * แสดงฟอร์มสร้าง Question Group
     */
    public function createQuestionGroup()
    {
        return view('admin.question-groups.create');
    }

    /**
     * บันทึก Question Group ใหม่
     */
    public function storeQuestionGroup(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['CusIns_ID'] = Auth::id();
        $validated['CusUdp_ID'] = Auth::id();
        $validated['created_by'] = Auth::id();
        $validated['type'] = 'manual';
        $validated['is_active'] = $request->has('is_active');

        QuestionGroup::create($validated);

        return redirect()->route('admin.question-groups')
            ->with('success', 'สร้างกลุ่มคำถามเรียบร้อยแล้ว');
    }

    /**
     * แสดงฟอร์มแก้ไข Question Group
     */
    public function editQuestionGroup(QuestionGroup $group)
    {
        // ดึงคำถามทั้งหมดและเรียงลำดับตาม sort_order
        $questions = $group->questions()
            ->whereNull('parent_id')
            ->with(['children' => function($query) {
                $query->orderBy('sort_order');
            }])
            ->orderBy('sort_order')
            ->get();

        return view('admin.question-groups.edit', compact('group', 'questions'));
    }

    /**
     * อัปเดต Question Group
     */
    public function updateQuestionGroup(Request $request, QuestionGroup $group)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['CusUdp_ID'] = Auth::id();
        $validated['is_active'] = $request->has('is_active');

        $group->update($validated);

        return redirect()->route('admin.question-groups')
            ->with('success', 'อัปเดตกลุ่มคำถามเรียบร้อยแล้ว');
    }

    /**
     * ลบ Question Group
     */
    public function deleteQuestionGroup(QuestionGroup $group)
    {
        try {
            // ลบคำตอบที่เกี่ยวข้องกับคำถามในกลุ่มนี้
            $questionIds = $group->questions()->pluck('id')->toArray();
            if (!empty($questionIds)) {
                \App\Models\Answer::whereIn('question_id', $questionIds)->delete();
            }
            
            // ลบคำถามทั้งหมดในกลุ่มนี้
            $group->questions()->delete();
            
            // ลบ Group
            $group->delete();
            
            return redirect()->route('admin.question-groups')
                ->with('success', 'ลบกลุ่มคำถามเรียบร้อยแล้ว');
        } catch (\Exception $e) {
            return redirect()->route('admin.question-groups')
                ->with('error', 'เกิดข้อผิดพลาดในการลบกลุ่มคำถาม: ' . $e->getMessage());
        }
    }

    /**
     * Toggle active status ของ Question Group
     */
    public function toggleQuestionGroupActive(Request $request, QuestionGroup $group)
    {
        try {
            $isActive = $request->input('is_active', false);
            $group->update(['is_active' => $isActive]);
            
            return response()->json([
                'success' => true,
                'message' => $isActive ? 'เปิดใช้งานกลุ่มคำถามเรียบร้อยแล้ว' : 'ปิดใช้งานกลุ่มคำถามเรียบร้อยแล้ว'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการเปลี่ยนแปลงสถานะ'
            ], 500);
        }
    }

    /**
     * แสดงฟอร์มสร้างคำถาม
     */
    public function createQuestion(QuestionGroup $group, Request $request)
    {
        $parentId = $request->query('parent_id');
        $parentQuestion = null;
        
        if ($parentId) {
            $parentQuestion = Question::find($parentId);
        }
        
        return view('admin.questions.create', compact('group', 'parentQuestion'));
    }

    /**
     * บันทึกคำถามใหม่
     */
    public function storeQuestion(Request $request, QuestionGroup $group)
    {
        $validated = $request->validate([
            'question_text' => 'required|string',
            'description' => 'nullable|string',
            'use_in_evaluation' => 'boolean',
            'keyword_tag' => 'nullable|string|max:255',
            'question_type' => 'nullable|in:yes_no,multiple_choice,text,',
            'choices' => 'nullable|array',
            'choices.*' => 'nullable|string|max:255',
            'parent_id' => 'nullable|exists:sas_questions,id',
        ]);

        $validated['question_group_id'] = $group->id;
        $validated['CusIns_ID'] = Auth::id();
        $validated['CusUdp_ID'] = Auth::id();
        $validated['use_in_evaluation'] = $request->has('use_in_evaluation');
        
        // จัดการ parent_id - ถ้าไม่มีหรือเป็นค่าว่างให้เป็น null
        if (empty($validated['parent_id'])) {
            $validated['parent_id'] = null;
        }
        
        // จัดการ question_type - ถ้าเป็นค่าว่างให้เป็น NULL
        if (empty($validated['question_type'])) {
            $validated['question_type'] = null;
        }
        
        // กำหนด use_in_evaluation ตามประเภทคำถาม
        if ($validated['question_type'] === 'multiple_choice' || $validated['question_type'] === 'text') {
            $validated['use_in_evaluation'] = false; // ไม่ใช้ในการประเมินเสมอ
        }
        // เฉพาะ yes_no และ null (ไม่มี) เท่านั้นที่สามารถใช้ในการประเมินได้
        
        // จัดการ choices สำหรับ multiple_choice
        if ($validated['question_type'] === 'multiple_choice') {
            $choices = $request->input('choices', []);
            $choices = array_filter($choices); // ลบค่าว่าง
            $validated['choices'] = !empty($choices) ? json_encode($choices) : null;
        } else {
            $validated['choices'] = null;
        }
        
        // ตรวจสอบเลขข้อซ้ำ
        $parsedQuestion = \App\Helpers\QuestionHelper::parseQuestionText($validated['question_text']);
        if ($parsedQuestion['question_number']) {
            if (\App\Helpers\QuestionHelper::isDuplicateQuestionNumber($parsedQuestion['question_number'], $group->id)) {
                return back()->withInput()->withErrors([
                    'question_text' => 'เลขข้อ ' . $parsedQuestion['question_number'] . ' ซ้ำกับคำถามอื่นในกลุ่มนี้'
                ]);
            }
        }
        
        // ตั้งค่า sort_order อัตโนมัติ
        if ($validated['parent_id']) {
            // ถ้าเป็นคำถามย่อย ให้เรียงตาม parent_id
            $maxSortOrder = $group->questions()
                ->where('parent_id', $validated['parent_id'])
                ->max('sort_order') ?? 0;
        } else {
            // ถ้าเป็นคำถามหลัก ให้เรียงตาม top-level
            $maxSortOrder = $group->questions()
                ->whereNull('parent_id')
                ->max('sort_order') ?? 0;
        }
        $validated['sort_order'] = $maxSortOrder + 1;

        Question::create($validated);

        return redirect()->route('admin.question-groups.edit', $group)
            ->with('success', 'สร้างคำถามเรียบร้อยแล้ว');
    }

    /**
     * แสดงฟอร์มแก้ไขคำถาม
     */
    public function editQuestion(Question $question)
    {
        return view('admin.questions.edit', compact('question'));
    }

    /**
     * อัปเดตคำถาม
     */
    public function updateQuestion(Request $request, Question $question)
    {
        $validated = $request->validate([
            'question_text' => 'required|string',
            'description' => 'nullable|string',
            'use_in_evaluation' => 'boolean',
            'keyword_tag' => 'nullable|string|max:255',
            'question_type' => 'nullable|in:yes_no,multiple_choice,text,',
            'choices' => 'nullable|array',
            'choices.*' => 'nullable|string|max:255',
        ]);

        $validated['CusUdp_ID'] = Auth::id();
        $validated['use_in_evaluation'] = $request->has('use_in_evaluation');
        
        // จัดการ question_type - ถ้าเป็นค่าว่างให้เป็น NULL
        if (empty($validated['question_type'])) {
            $validated['question_type'] = null;
        }
        
        // กำหนด use_in_evaluation ตามประเภทคำถาม
        if ($validated['question_type'] === 'multiple_choice' || $validated['question_type'] === 'text') {
            $validated['use_in_evaluation'] = false; // ไม่ใช้ในการประเมินเสมอ
        }
        
        // จัดการ choices สำหรับ multiple_choice
        if ($validated['question_type'] === 'multiple_choice') {
            $choices = $request->input('choices', []);
            $choices = array_filter($choices); // ลบค่าว่าง
            $validated['choices'] = !empty($choices) ? json_encode($choices) : null;
        } else {
            $validated['choices'] = null;
        }

        // ตรวจสอบเลขข้อซ้ำ (ยกเว้นคำถามปัจจุบัน)
        $parsedQuestion = \App\Helpers\QuestionHelper::parseQuestionText($validated['question_text']);
        if ($parsedQuestion['question_number']) {
            if (\App\Helpers\QuestionHelper::isDuplicateQuestionNumber($parsedQuestion['question_number'], $question->question_group_id, $question->id)) {
                return back()->withInput()->withErrors([
                    'question_text' => 'เลขข้อ ' . $parsedQuestion['question_number'] . ' ซ้ำกับคำถามอื่นในกลุ่มนี้'
                ]);
            }
        }

        $question->update($validated);

        return redirect()->route('admin.question-groups.edit', $question->questionGroup)
            ->with('success', 'อัปเดตคำถามเรียบร้อยแล้ว');
    }

    /**
     * ลบคำถาม
     */
    public function deleteQuestion(Question $question)
    {
        try {
            $group = $question->questionGroup;
            $question->delete();
            
            return redirect()->route('admin.question-groups.edit', $group)
                ->with('success', 'ลบคำถามเรียบร้อยแล้ว');
        } catch (\Exception $e) {
            return redirect()->route('admin.question-groups.edit', $question->questionGroup)
                ->with('error', 'ไม่สามารถลบคำถามได้ เนื่องจากมีการใช้งานอยู่');
        }
    }

    /**
     * อัปเดตลำดับคำถาม (Drag & Drop)
     */
    public function updateQuestionOrder(Request $request)
    {
        try {
            $request->validate([
                'questions' => 'required|array',
                'questions.*.id' => 'required|exists:sas_questions,id',
                'questions.*.parent_id' => 'nullable|exists:sas_questions,id',
            ]);

            foreach ($request->questions as $questionData) {
                Question::where('id', $questionData['id'])->update([
                    'parent_id' => $questionData['parent_id'] ?? null,
                    'CusUdp_ID' => Auth::id(),
                ]);
            }

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการอัปเดตความสัมพันธ์คำถาม: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * อัปเดต logic operator ของคำถามแม่
     */
    public function updateQuestionLogic(Request $request, Question $question)
    {
        $request->validate([
            'logic_operator' => 'required|in:AND,OR',
        ]);

        try {
            // ตรวจสอบว่าคำถามนี้มีคำถามย่อยหรือไม่
            if ($question->children()->count() == 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'คำถามนี้ไม่มีคำถามย่อย'
                ], 400);
            }

            // อัปเดต logic operator
            $question->update([
                'logic_operator' => $request->logic_operator,
                'CusUdp_ID' => Auth::id(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'อัปเดตเงื่อนไขการเชื่อมต่อเรียบร้อยแล้ว'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการอัปเดตเงื่อนไขการเชื่อมต่อ'
            ], 500);
        }
    }

    /**
     * เปลี่ยนตำแหน่งคำถาม (ขึ้น/ลง)
     */
    public function moveQuestion(Request $request)
    {
        $request->validate([
            'question_id' => 'required|exists:sas_questions,id',
            'direction' => 'required|in:up,down',
        ]);

        try {
            $question = Question::findOrFail($request->question_id);
            $direction = $request->direction;

            // หาคำถามที่อยู่ในกลุ่มเดียวกัน (parent_id เดียวกัน)
            $siblings = Question::where('question_group_id', $question->question_group_id)
                ->where('parent_id', $question->parent_id)
                ->orderBy('sort_order')
                ->get();

            $currentIndex = $siblings->search(function($item) use ($question) {
                return $item->id == $question->id;
            });

            if ($direction === 'up' && $currentIndex > 0) {
                // เลื่อนขึ้น
                $targetQuestion = $siblings[$currentIndex - 1];
                $tempSortOrder = $question->sort_order;
                
                $question->update(['sort_order' => $targetQuestion->sort_order, 'CusUdp_ID' => Auth::id()]);
                $targetQuestion->update(['sort_order' => $tempSortOrder, 'CusUdp_ID' => Auth::id()]);
                
            } elseif ($direction === 'down' && $currentIndex < $siblings->count() - 1) {
                // เลื่อนลง
                $targetQuestion = $siblings[$currentIndex + 1];
                $tempSortOrder = $question->sort_order;
                
                $question->update(['sort_order' => $targetQuestion->sort_order, 'CusUdp_ID' => Auth::id()]);
                $targetQuestion->update(['sort_order' => $tempSortOrder, 'CusUdp_ID' => Auth::id()]);
                
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'ไม่สามารถเลื่อนคำถามได้'
                ], 400);
            }

            return response()->json([
                'success' => true,
                'message' => 'เปลี่ยนตำแหน่งคำถามเรียบร้อยแล้ว'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'เกิดข้อผิดพลาดในการเปลี่ยนตำแหน่งคำถาม: ' . $e->getMessage()
            ], 500);
        }
    }



    /**
     * แสดงแผนผังโครงสร้างคำถามด้วย D3.js
     */
    public function showQuestionTreeStructure(QuestionGroup $group)
    {
        // ✅ ใช้ QuestionHelper แทนการเรียก method ที่ไม่มี
        $d3Data = QuestionHelper::questionGroupToD3($group);
        
        return view('admin.question-groups.d3-tree-structure', compact('group', 'd3Data'));
    }
} 