<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Assessment;
use App\Models\Answer;
use App\Models\Question;
use App\Models\QuestionGroup;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class AssessmentController extends Controller
{
    /* ========= Shared helpers ========= */

    /** ตรวจสิทธิ์กับ Product */
    private function authorizeProduct(Product $product): void
    {
        if (Auth::user()->Customer_ID !== $product->user_id && Auth::user()->Level !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }
    }

    /** ตรวจสิทธิ์กับ Assessment */
    private function authorizeAssessment(Assessment $assessment): void
    {
        if (Auth::user()->Customer_ID !== $assessment->user_id && Auth::user()->Level !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }
    }

    /** ดึง QuestionGroup ที่ active ตัวแรก (กัน null) */
    private function getActiveQuestionGroupOrFail(): QuestionGroup
    {
        $group = QuestionGroup::active()->first();
        if (!$group) abort(422, 'ยังไม่มีชุดคำถามที่เปิดใช้งาน');
        return $group;
    }

    /** แปลงคำตอบจาก form ให้เป็นข้อความ + comment */
    private function normalizeAnswer(array $answerData): array
    {
        $answerText = null;
        $comment = null;

        if (isset($answerData['choice'])) {
            $choice = $answerData['choice'];
            if ($choice === 'yes')       $answerText = 'ใช่';
            elseif ($choice === 'no')    $answerText = 'ไม่ใช่';
            else                         $answerText = $choice; // multiple_choice
        } elseif (isset($answerData['text'])) {
            $answerText = $answerData['text'];
        }

        if (isset($answerData['details'])) $comment = $answerData['details'];

        return [$answerText, $comment];
    }

    /** บันทึกคำตอบของคำถามลูกทั้งหมด (ข้ามคำถามที่ question_type = null) */
    private function saveChildAnswers(Assessment $assessment, array $answersForm): void
    {
        foreach ($answersForm as $qid => $data) {
            $questionId = (int)$qid;
            $question   = Question::find($questionId);
            if (!$question) {
                Log::warning('Question not found', ['question_id' => $questionId]);
                continue;
            }
            // เก็บเฉพาะคำถามที่เป็นชนิดจริง (ไม่ใช่ parent)
            if ($question->question_type === null) continue;

            [$answerText, $comment] = $this->normalizeAnswer($data);
            if (!$answerText) continue;

            Answer::create([
                'assessment_id' => $assessment->id,
                'question_id'   => $questionId,
                'answer_text'   => $answerText,
                'image_path'    => null,
                'comment'       => $comment,
                'CusIns_ID'     => Auth::user()->Customer_ID,
                'CusUdp_ID'     => Auth::user()->Customer_ID,
            ]);

            Log::info('Answer saved', ['question_id' => $questionId, 'answer_text' => $answerText]);
        }
    }

    /** คำนวณผลคำถาม parent (recursive) และบันทึกเป็น Answer ของ parent */
    private function computeAndSaveParentResults(Assessment $assessment): void
    {
        $parentQuestions = Question::whereNull('question_type')
            ->where('use_in_evaluation', 1)
            ->whereHas('questionGroup', fn($q) => $q->where('is_active', true))
            ->get();

        $calc = function ($parentId) use (&$calc, $assessment): bool {
            $children = Question::where('parent_id', $parentId)->where('use_in_evaluation', 1)->get();
            if ($children->isEmpty()) return false;

            $flags = [];
            foreach ($children as $child) {
                if ($child->question_type === 'yes_no') {
                    $ans = $assessment->answers()->where('question_id', $child->id)->first();
                    $flags[] = $ans ? ($ans->answer_text === 'ใช่') : false;
                } elseif ($child->question_type === null) {
                    $flags[] = $calc($child->id);
                }
            }

            $parent = Question::find($parentId);
            if (!$parent) return false;

            return match ($parent->logic_operator) {
                'AND' => !in_array(false, $flags, true),
                'OR'  => in_array(true, $flags, true),
                default => false,
            };
        };

        foreach ($parentQuestions as $parent) {
            $result = $calc($parent->id);
            Answer::create([
                'assessment_id' => $assessment->id,
                'question_id'   => $parent->id,
                'answer_text'   => $result ? 'ใช่' : 'ไม่ใช่',
                'image_path'    => null,
                'comment'       => null,
                'CusIns_ID'     => Auth::user()->Customer_ID,
                'CusUdp_ID'     => Auth::user()->Customer_ID,
            ]);
            Log::info('Parent answer saved', [
                'question_id' => $parent->id,
                'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                'logic'       => $parent->logic_operator,
            ]);
        }
    }

    /** ลบคำตอบเดิมทั้งหมดของ assessment */
    private function clearAnswers(Assessment $assessment): void
    {
        $deleted = $assessment->answers()->delete();
        Log::info('Cleared answers', ['assessment_id' => $assessment->id, 'count' => $deleted]);
    }

    /* ========= Actions ========= */

    public function create(Product $product)
    {
        $this->authorizeProduct($product);
        $questionGroups = QuestionGroup::active()->with('questions')->get();
        return view('assessments.create', compact('product', 'questionGroups'));
    }

    /** สร้าง Product (จาก session) + Assessment (submitted) และบันทึกคำตอบ */
    public function store(Request $request)
    {
        Log::info('Assessment Store Request', [
            'answers' => $request->answers,
            'session_pending_product' => session('pending_product'),
        ]);

        $request->validate([
            'answers' => 'required|array',
            'answers.*.choice'  => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text'    => 'nullable|string',
        ]);

        $pending = session('pending_product');
        if (!$pending) {
            return redirect()->route('products.create')->with('error', 'กรุณากรอกข้อมูลผลิตภัณฑ์ก่อน');
        }

        try {
            return DB::transaction(function () use ($pending, $request) {
                // สร้าง Product
                $pending['user_id']   = Auth::user()->Customer_ID;
                $pending['CusIns_ID'] = Auth::user()->Customer_ID;
                $pending['CusUdp_ID'] = Auth::user()->Customer_ID;
                $product = Product::create($pending);
                $request->session()->forget('pending_product');
                Log::info('Product created', ['product_id' => $product->id]);

                // สร้าง Assessment (submitted)
                $group = $this->getActiveQuestionGroupOrFail();
                $assessment = Assessment::create([
                    'product_id'       => $product->id,
                    'question_group_id'=> $group->id,
                    'user_id'          => Auth::user()->Customer_ID,
                    'CusIns_ID'        => Auth::user()->Customer_ID,
                    'CusUdp_ID'        => Auth::user()->Customer_ID,
                    'status'           => 'submitted',
                    'started_at'       => now(),
                    'submitted_at'     => now(),
                ]);
                Log::info('Assessment created', ['assessment_id' => $assessment->id]);

                // บันทึกคำตอบ
                $this->clearAnswers($assessment);
                $this->saveChildAnswers($assessment, $request->answers);
                $this->computeAndSaveParentResults($assessment);

                return redirect()->route('products.report', $product)
                    ->with('success', 'บันทึกการประเมินเรียบร้อยแล้ว');
            });
        } catch (\Throwable $e) {
            Log::error('Assessment store error', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return back()->withInput()->with('error', 'เกิดข้อผิดพลาดในการบันทึกการประเมิน: '.$e->getMessage());
        }
    }

    /** ส่งแบบประเมินจากฉบับร่าง */
    public function submitDraft(Request $request)
    {
        if (!$request->has('product_id')) {
            return back()->with('error', 'ไม่พบข้อมูลผลิตภัณฑ์');
        }

        $assessment = Assessment::where('product_id', $request->product_id)
            ->where('status', 'draft')
            ->where('user_id', Auth::user()->Customer_ID)
            ->first();

        if (!$assessment) return back()->with('error', 'ไม่พบฉบับร่างหรือไม่มีสิทธิ์เข้าถึง');
        $this->authorizeAssessment($assessment);

        $request->validate([
            'answers' => 'required|array',
            'answers.*.choice'  => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text'    => 'nullable|string',
        ]);

        try {
            return DB::transaction(function () use ($assessment, $request) {
                // เปลี่ยนเป็น submitted
                $assessment->update([
                    'status'       => 'submitted',
                    'submitted_at' => now(),
                    'CusUdp_ID'    => Auth::user()->Customer_ID,
                    'DtmUpd'       => now(),
                ]);

                // บันทึกคำตอบใหม่
                $this->clearAnswers($assessment);
                $this->saveChildAnswers($assessment, $request->answers);
                $this->computeAndSaveParentResults($assessment);

                // ล้าง session pending_product (ถ้ามี)
                session()->forget('pending_product');

                $product = Product::find($assessment->product_id);
                if (!$product) return redirect()->route('products.index')->with('error', 'ไม่พบข้อมูลผลิตภัณฑ์');

                return redirect()->route('products.report', $product)->with('success', 'ส่งแบบประเมินเรียบร้อยแล้ว');
            });
        } catch (\Throwable $e) {
            Log::error('submitDraft error', ['message' => $e->getMessage()]);
            return back()->with('error', 'เกิดข้อผิดพลาดในการส่งแบบประเมิน: '.$e->getMessage());
        }
    }

    /** บันทึกฉบับร่างการประเมิน (มีสร้าง Product ถ้ามาจาก session) */
    public function storeDraft(Request $request)
    {
        Log::info('Assessment Draft Store Request', [
            'answers' => $request->answers,
            'session_pending_product' => session('pending_product'),
        ]);

        $request->validate([
            'answers' => 'nullable|array',
            'answers.*.choice'  => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text'    => 'nullable|string',
        ]);

        // หา/สร้าง Product
        $product = null;
        $assessment = null;

        if (session('pending_product')) {
            $pending = session('pending_product');
            $pending['user_id']   = Auth::user()->Customer_ID;
            $pending['CusIns_ID'] = Auth::user()->Customer_ID;
            $pending['CusUdp_ID'] = Auth::user()->Customer_ID;
            try {
                $product = DB::transaction(function () use ($pending) {
                    $p = Product::create($pending);
                    session()->forget('pending_product');
                    return $p;
                });
                Log::info('Product created for draft', ['product_id' => $product->id]);
            } catch (\Throwable $e) {
                return back()->with('error', 'สร้างผลิตภัณฑ์ไม่สำเร็จ: '.$e->getMessage());
            }
        } else {
            if ($pid = $request->input('product_id')) $product = Product::find($pid);
            if ($product) {
                $assessment = Assessment::where('product_id', $product->id)->where('status', 'draft')->first();
            }
        }

        if (!$product) {
            return redirect()->route('products.create')->with('error', 'กรุณากรอกข้อมูลผลิตภัณฑ์ก่อน');
        }

        try {
            return DB::transaction(function () use ($request, $product, $assessment) {
                // สร้างหรืออัปเดต draft
                if (!$assessment) {
                    $group = $this->getActiveQuestionGroupOrFail();
                    $assessment = Assessment::create([
                        'product_id'        => $product->id,
                        'question_group_id' => $group->id,
                        'user_id'           => Auth::user()->Customer_ID,
                        'CusIns_ID'         => Auth::user()->Customer_ID,
                        'CusUdp_ID'         => Auth::user()->Customer_ID,
                        'status'            => 'draft',
                        'started_at'        => now(),
                        'submitted_at'      => null,
                        'DtmUpd'            => now(),
                    ]);
                    Log::info('New draft created', ['assessment_id' => $assessment->id]);
                } else {
                    $assessment->update([
                        'CusUdp_ID' => Auth::user()->Customer_ID,
                        'DtmUpd'    => now(),
                    ]);
                    Log::info('Draft updated', ['assessment_id' => $assessment->id]);
                }

                // เขียนคำตอบ (ถ้ามี)
                $this->clearAnswers($assessment);
                if ($request->has('answers') && is_array($request->answers)) {
                    $this->saveChildAnswers($assessment, $request->answers);
                }
                // คำนวณ parent เพื่อให้ draft แสดงผลสรุปเบื้องต้นได้
                $this->computeAndSaveParentResults($assessment);

                return redirect()
                    ->route('products.index')
                    ->with('success', 'บันทึกฉบับร่างเรียบร้อยแล้ว คุณสามารถกด "ทำต่อ" ได้');
            });
        } catch (\Throwable $e) {
            Log::error('storeDraft error', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return back()->withInput()->with('error', 'เกิดข้อผิดพลาดในการบันทึกฉบับร่าง: '.$e->getMessage());
        }
    }

    /** แสดงฟอร์มทำต่อจากฉบับร่าง */
    public function continueDraft(Product $product)
    {
        $this->authorizeProduct($product);

        $assessment = Assessment::where('product_id', $product->id)->where('status', 'draft')->first();
        if (!$assessment) {
            return redirect()->route('products.index')->with('error', 'ไม่พบฉบับร่างสำหรับผลิตภัณฑ์นี้');
        }

        $questionGroups = QuestionGroup::active()->with('questions')->get();
        $assessment->load(['answers' => fn($q) => $q->orderBy('question_id')]);

        $validated = [
            'name'              => $product->name,
            'model'             => $product->model,
            'product_type'      => $product->product_type,
            'log_receive_method'=> $product->log_receive_method,
            'description'       => $product->description,
        ];

        $existingAnswers = [];
        foreach ($assessment->answers as $ans) {
            $existingAnswers[$ans->question_id] = [
                'choice'  => $ans->answer_text,
                'details' => $ans->comment,
                'text'    => $ans->answer_text,
            ];
        }

        return view('assessments.manual', compact('assessment', 'questionGroups', 'validated', 'existingAnswers'));
    }

    /** หน้าแก้ไข (submitted) */
    public function edit(Assessment $assessment)
    {
        $this->authorizeAssessment($assessment);
        Log::info('User level: '.Auth::user()->Level);
        Log::info('User Customer_ID: '.Auth::user()->Customer_ID);
        Log::info('Assessment user_id: '.$assessment->user_id);

        $questionGroups = QuestionGroup::active()->with('questions')->get();
        return view('assessments.edit', compact('assessment', 'questionGroups'));
    }

    /** หน้าแบบประเมินคู่มือ (ต้องเป็น draft เท่านั้น) */
    public function manual(Assessment $assessment)
    {
        $this->authorizeAssessment($assessment);
        if ($assessment->status !== 'draft') {
            return redirect()->route('assessments.edit', $assessment)->with('error', 'ไม่สามารถเข้าถึงหน้าแบบประเมินคู่มือได้');
        }

        $questionGroups = QuestionGroup::active()->with('questions')->get();
        $assessment->load('answers');

        return view('assessments.manual', compact('assessment', 'questionGroups'));
    }

    /** อัปเดต (ส่ง) การประเมินที่แก้ไข */
    public function update(Request $request, Assessment $assessment)
    {
        $this->authorizeAssessment($assessment);

        $request->validate([
            'answers' => 'nullable|array',
            'answers.*.choice'  => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text'    => 'nullable|string',
        ]);

        try {
            return DB::transaction(function () use ($assessment, $request) {
                $assessment->update([
                    'status'       => 'submitted',
                    'submitted_at' => now(),
                    'CusUdp_ID'    => Auth::user()->Customer_ID,
                ]);

                $this->clearAnswers($assessment);
                if ($request->has('answers') && is_array($request->answers)) {
                    $this->saveChildAnswers($assessment, $request->answers);
                }
                $this->computeAndSaveParentResults($assessment);

                return redirect()->route('products.report', $assessment->product)
                    ->with('success', 'ส่งแบบประเมินเรียบร้อยแล้ว');
            });
        } catch (\Throwable $e) {
            Log::error('update error', ['message' => $e->getMessage()]);
            return back()->withInput()->with('error', 'เกิดข้อผิดพลาดในการส่งแบบประเมิน: '.$e->getMessage());
        }
    }
}
