<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

class Product extends Model
{
    use HasFactory;

    /**
     * ฟิลด์ที่อนุญาตให้กรอกผ่าน mass assignment
     */
    protected $fillable = [
        'name',
        'model',
        'log_receive_method',
        'product_type',
        'description',
        'image',
        'user_id',
        'CusIns_ID',
        'CusUdp_ID',
        // แนะนำให้เพิ่มคอลัมน์นี้ใน DB (ถ้ายังไม่มี ให้ดู migration ที่คุยกันก่อนหน้า)
        'agency_id',
    ];

    // ใช้ชื่อคอลัมน์เวลาแบบกำหนดเอง
    const CREATED_AT = 'DtmIns';
    const UPDATED_AT = 'DtmUpd';

    /**
     * ความสัมพันธ์: เจ้าของผลิตภัณฑ์ (ตาราง customer)
     */
    public function user()
    {
        return $this->belongsTo(Customer::class, 'user_id', 'Customer_ID');
    }

    /**
     * ความสัมพันธ์: แบบประเมินของผลิตภัณฑ์นี้
     */
    public function assessments()
    {
        return $this->hasMany(Assessment::class);
    }

    /**
     * (ถ้ามีโมเดล Agency) ความสัมพันธ์ไปยังบริษัท
     * รองรับทั้งสคีมา 'agency(Agency_ID)' และ 'agencies(id)'
     */
    public function agency()
    {
        if (Schema::hasTable('agency')) {
            return $this->belongsTo(Agency::class, 'agency_id', 'Agency_ID');
        }
        return $this->belongsTo(Agency::class, 'agency_id', 'id');
    }

    /**
     * แปลง label ประเภทสินค้า
     */
    public function getProductTypeLabelAttribute()
    {
        return match ($this->product_type) {
            'hw_sw_log_server' => 'Hardware + Software Log Server',
            'sw_log_server'    => 'Software Log Server',
            default            => $this->product_type,
        };
    }

    /**
     * แปลง label วิธีรับ log
     */
    public function getLogReceiveMethodLabelAttribute()
    {
        return match ($this->log_receive_method) {
            'syslog_udp'  => 'syslog UDP',
            'syslog_tcp'  => 'syslog TCP',
            'syslog_both' => 'syslog UDP และ syslog TCP',
            'other'       => 'อื่นๆ',
            default       => $this->log_receive_method,
        };
    }

    /* ===================== Multi-tenant by Agency ===================== */

    /**
     * Global scope: บังคับทุก query จำกัดตาม Agency ปัจจุบันเสมอ
     * - ถ้ามีคอลัมน์ products.agency_id จะ where ตรงๆ
     * - ถ้าไม่มี จะ fallback ผ่าน owner (customer.Agency_ID)
     */
    protected static function booted(): void
    {
        static::addGlobalScope('agency', function (Builder $q) {
            $aid = (int) session('current_agency_id');
            if ($aid > 0) {
                if (Schema::hasColumn('products', 'agency_id')) {
                    $q->where('agency_id', $aid);
                } else {
                    $q->whereHas('user', fn ($u) => $u->where('Agency_ID', $aid));
                }
            }
        });

        // ตอนสร้าง: ถ้ามีคอลัมน์ agency_id ให้ติดค่านั้นอัตโนมัติจาก session
        static::creating(function (Product $p) {
            if (Schema::hasColumn('products', 'agency_id')) {
                $aid = (int) session('current_agency_id');
                abort_if($aid <= 0, 403, 'No agency context');
                // กันการยัดค่า cross-agency: บังคับทับให้ตรงกับ session เสมอ
                $p->agency_id = $aid;
            }
        });

        // (ปลอดภัยเพิ่ม) ตอนอัปเดต: กันการย้ายข้าม agency โดยไม่ตั้งใจ
        static::updating(function (Product $p) {
            if (Schema::hasColumn('products', 'agency_id')) {
                $aid = (int) session('current_agency_id');
                abort_if($aid <= 0, 403, 'No agency context');
                if ((int) $p->agency_id !== $aid) {
                    $p->agency_id = $aid;
                }
            }
        });
    }

    /**
     * (เผื่อใช้) scope สำหรับระบุ agency แบบชัดเจน โดยข้าม global scope
     */
    public function scopeForAgency(Builder $q, int $aid)
    {
        return Schema::hasColumn('products', 'agency_id')
            ? $q->withoutGlobalScope('agency')->where('agency_id', $aid)
            : $q->withoutGlobalScope('agency')->whereHas('user', fn ($u) => $u->where('Agency_ID', $aid));
    }
}
