<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class QuestionGroup extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'is_active',
        'type',
        'CusIns_ID',
        'CusUdp_ID',
        'created_by'
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    // กำหนดชื่อ timestamp columns
    const CREATED_AT = 'DtmIns';
    const UPDATED_AT = 'DtmUpd';

    /**
     * ความสัมพันธ์: QuestionGroup มีหลาย Question
     */
    public function questions()
    {
        return $this->hasMany(Question::class);
    }

    /**
     * ความสัมพันธ์กับผู้สร้าง (Customer)
     */
    public function creator()
    {
        return $this->belongsTo(Customer::class, 'CusIns_ID', 'Customer_ID');
    }

    /**
     * ความสัมพันธ์กับผู้แก้ไข (Customer)
     */
    public function updater()
    {
        return $this->belongsTo(Customer::class, 'CusUdp_ID', 'Customer_ID');
    }

    /**
     * ความสัมพันธ์กับ Assessment (ถ้ามีการใช้ group นี้ในแบบประเมิน)
     */
    public function assessments()
    {
        return $this->hasMany(Assessment::class);
    }

    /**
     * Accessor สำหรับชื่อกลุ่ม
     */
    public function getTitleAttribute($value)
    {
        return $this->name;
    }

    /**
     * ตรวจสอบว่ากลุ่มนี้ถูกใช้งานหรือไม่
     */
    public function isInUse()
    {
        return $this->assessments()->count() > 0;
    }

    /**
     * จำนวนคำถามทั้งหมดในกลุ่ม
     */
    public function getTotalQuestionsAttribute()
    {
        return $this->questions()->count();
    }

    /**
     * จำนวนคำถามที่ใช้ในการประเมิน
     */
    public function getEvaluationQuestionsAttribute()
    {
        return $this->questions()->where('use_in_evaluation', true)->count();
    }

    /**
     * Scope สำหรับดึงเฉพาะกลุ่มที่ active
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * ดึงคำถามทั้งหมดที่ใช้ในการประเมินจากกลุ่มที่ active
     */
    public function getActiveEvaluationQuestions()
    {
        return $this->questions()
            ->where('use_in_evaluation', true)
            ->whereNotNull('question_type')
            ->whereNull('parent_id')
            ->with(['children' => function($query) {
                $query->where('use_in_evaluation', true)
                      ->whereNotNull('question_type')
                      ->orderBy('sort_order');
            }])
            ->orderBy('sort_order')
            ->get();
    }
}