@extends('layouts.admin')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header bg-gradient-primary text-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h4 class="mb-0">
                                <i class="fas fa-project-diagram me-2"></i>
                                แผนผังโครงสร้างคำถาม
                            </h4>
                            <p class="mb-0 mt-1 opacity-75">
                                <i class="fas fa-info-circle me-1"></i>
                                {{ $group->name }}
                            </p>
                        </div>
                        <div class="d-flex gap-2">
                            <button class="btn btn-light btn-sm" onclick="toggleFullscreen()">
                                <i class="fas fa-expand me-1"></i>
                                เต็มหน้าจอ
                            </button>
                            <a href="{{ route('admin.question-groups') }}" class="btn btn-light btn-sm">
                                <i class="fas fa-arrow-left me-1"></i>
                                กลับไปหน้าจัดการคำถาม
                            </a>
                        </div>
                    </div>
                </div>
                
                <div class="card-body p-0">
                    <!-- Legend -->
                    <div class="bg-light p-3 border-bottom">
                        <div class="row">
                            <div class="col-md-6">
                                <h6 class="mb-2"><i class="fas fa-info-circle me-1"></i>คำอธิบายสัญลักษณ์</h6>
                                <div class="d-flex flex-wrap gap-3">
                                    <div class="d-flex align-items-center">
                                        <div class="legend-icon and-node me-2"></div>
                                        <small>คำถามแม่ (Logic AND)</small>
                                    </div>
                                    <div class="d-flex align-items-center">
                                        <div class="legend-icon or-node me-2"></div>
                                        <small>คำถามแม่ (Logic OR)</small>
                                    </div>
                                    <div class="d-flex align-items-center">
                                        <div class="legend-icon leaf-node me-2"></div>
                                        <small>คำถามย่อย</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6 class="mb-2"><i class="fas fa-mouse-pointer me-1"></i>วิธีใช้งาน</h6>
                                <div class="d-flex flex-wrap gap-3">
                                    <small><i class="fas fa-search-plus me-1"></i>Scroll เพื่อ Zoom</small>
                                    <small><i class="fas fa-arrows-alt me-1"></i>ลากเพื่อเลื่อน</small>
                                    <small><i class="fas fa-mouse me-1"></i>คลิกเพื่อดูรายละเอียด</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- D3.js Chart Container -->
                    <div id="org-chart-container">
                        <div id="org-chart"></div>
                        
                        <!-- Loading Indicator -->
                        <div id="loading" class="text-center py-5">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">กำลังโหลด...</span>
                            </div>
                            <p class="mt-2 text-muted">กำลังสร้างแผนผังโครงสร้างคำถาม...</p>
                        </div>
                        
                        <!-- Error Message -->
                        <div id="error" class="text-center py-5" style="display: none;">
                            <i class="fas fa-exclamation-triangle fa-3x text-warning mb-3"></i>
                            <h5 class="text-warning">เกิดข้อผิดพลาด</h5>
                            <p class="text-muted" id="error-message">ไม่สามารถสร้างแผนผังได้</p>
                            <button class="btn btn-outline-warning" onclick="retryChart()">
                                <i class="fas fa-redo me-1"></i>
                                ลองใหม่
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
</div>

<!-- D3.js Script -->
<script>
// ข้อมูลจาก Controller
const chartData = @json($d3Data ?? []);

// D3.js Family Tree Chart
document.addEventListener('DOMContentLoaded', function() {
    if (chartData && chartData.length > 0) {
        createFamilyTree(chartData);
    } else {
        showError('ไม่พบข้อมูลคำถามในกลุ่มนี้');
    }
});

function createFamilyTree(data) {
    try {
        // ซ่อน loading
        document.getElementById('loading').style.display = 'none';
        
        if (!data || data.length === 0) {
            showError('ไม่พบข้อมูลคำถาม');
            return;
        }

        // ตั้งค่าขนาดสำหรับ family tree
        const width = 1600; // ขยายความกว้างจาก 1200 เป็น 1600
        const height = 1200;
        
        // สร้าง tree layout แนวตั้ง (เหมือน family tree)
        const tree = d3.tree()
            .size([width - 200, 600]) // ลดความสูงจาก height - 200 เป็น 600 เพื่อให้เส้นในแนวแกน Y ไม่ยาวเกินไป
            .separation((a, b) => (a.parent == b.parent ? 16.0 : 20.0)); // เพิ่มระยะห่างระหว่างโหนดที่ parent คนละตัว
        
        // สร้าง hierarchy
        const root = d3.hierarchy(data[0]); // ใช้คำถามแรกเป็น root
        const nodes = tree(root);
        
        // สร้าง SVG
        const svg = d3.select("#org-chart")
            .append("svg")
            .attr("width", width)
            .attr("height", height)
            .attr("viewBox", [0, 0, width, height])
            .style("font", "14px sans-serif");
        
        // สร้าง container group สำหรับ zoom
        const container = svg.append("g")
            .attr("transform", "translate(100, 100)");
        
        // เพิ่ม zoom functionality
        const zoom = d3.zoom()
            .scaleExtent([0.3, 3])
            .on("zoom", (event) => {
                container.attr("transform", event.transform);
            });
        
        svg.call(zoom);
        
        // วาดเส้นเชื่อมแบบหักมุม 90 องศา (เหมือนรูปที่ส่งมา)
        container.selectAll("path")
            .data(nodes.descendants().slice(1))
            .join("path")
            .attr("fill", "none")
            .attr("stroke", "#6c757d")
            .attr("stroke-width", 2)
            .attr("stroke-opacity", 0.6)
            .attr("d", d => {
                // เส้นหักมุม 90 องศาแบบรูปตัว L หรือตัว T คว่ำ
                const midY = (d.y + d.parent.y) / 2;
                return `M${d.x},${d.y} 
                        L${d.x},${midY}
                        L${d.parent.x},${midY}
                        L${d.parent.x},${d.parent.y}`;
            });
        
        // วาดโหนด
        const node = container.selectAll("g")
            .data(nodes.descendants())
            .join("g")
            .attr("transform", d => `translate(${d.x},${d.y})`)
            .style("cursor", "default") // เปลี่ยนจาก pointer เป็น default
            .on("mouseover", function(event, d) {
                d3.select(this).select("circle").attr("r", d.children ? 15 : 12);
                d3.select(this).select("text").style("font-weight", "bold");
            })
            .on("mouseout", function(event, d) {
                d3.select(this).select("circle").attr("r", d.children ? 12 : 10);
                d3.select(this).select("text").style("font-weight", "normal");
            });
        
        // สร้างโหนดแบบวงกลม (เหมือนในคำอธิบายสัญลักษณ์)
        node.append("circle")
            .attr("r", d => d.children ? 12 : 10)
            .attr("fill", d => getNodeColor(d.data))
            .attr("stroke", "#fff")
            .attr("stroke-width", 3)
            .attr("class", d => d.children ? "node-expandable" : "node-leaf")
            .style("filter", "drop-shadow(0 2px 4px rgba(0,0,0,0.2))");
        
        // แสดงเลขข้อข้างโหนด (ด้านขวา)
        node.append("text")
            .attr("dy", "0.35em")
            .attr("x", d => d.children ? 18 : 16) // อยู่ด้านขวาของโหนด
            .attr("text-anchor", "start")
            .text(d => d.data.question_number || '')
            .style("font-size", "12px")
            .style("font-weight", "bold")
            .style("fill", "#2c3e50")
            .attr("class", "node-number")
            .style("text-shadow", "1px 1px 2px rgba(255,255,255,0.8)");
        
        // เพิ่ม tooltip
        node.append("title")
            .text(d => `${d.data.question_number || ''} ${d.data.name}`);
        
        console.log('Family Tree created successfully');
        
    } catch (error) {
        console.error('Error creating family tree:', error);
        showError('เกิดข้อผิดพลาดในการสร้างแผนผัง: ' + error.message);
    }
}

// ฟังก์ชันช่วยเหลือ
function getNodeColor(data) {
    if (data.children && data.children.length > 0) {
        if (data.logic_operator === 'OR') {
            return '#e74c3c'; // สีแดงสำหรับ OR
        } else {
            return '#27ae60'; // สีเขียวสำหรับ AND
        }
    }
    return '#3498db'; // สีน้ำเงินสำหรับคำถามย่อย
}

function truncateText(text, maxLength) {
    if (text.length <= maxLength) return text;
    return text.substring(0, maxLength) + '...';
}

function toggleFullscreen() {
    const container = document.getElementById('org-chart-container');
    if (!document.fullscreenElement) {
        container.requestFullscreen();
    } else {
        document.exitFullscreen();
    }
}

function showError(message) {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('error').style.display = 'block';
    document.getElementById('error-message').textContent = message;
}

function retryChart() {
    document.getElementById('error').style.display = 'none';
    document.getElementById('loading').style.display = 'block';
    document.getElementById('org-chart').innerHTML = '';
    createFamilyTree(chartData);
}
</script>

<style>
.bg-gradient-primary {
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
}

.card.shadow {
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
}

.legend-icon {
    width: 24px;
    height: 24px;
    border-radius: 50%;
    border: 3px solid #fff;
    box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}

.and-node {
    background: #27ae60;
}

.or-node {
    background: #e74c3c;
}

.leaf-node {
    background: #3498db;
}

#org-chart-container {
    background: white;
    min-height: 800px;
    position: relative;
}

#org-chart {
    text-align: center;
    overflow: hidden;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
}

#org-chart svg {
    max-width: 100%;
    height: auto;
    filter: drop-shadow(0 4px 8px rgba(0,0,0,0.1));
}

/* ป้องกันข้อความและเลขข้อถูกยุบเมื่อซูม */
.node-number {
    pointer-events: none;
    transform-origin: center;
}

.node-expandable {
    transition: all 0.3s ease;
}

.node-expandable:hover {
    filter: drop-shadow(0 4px 8px rgba(0,0,0,0.3));
}

.node-leaf {
    transition: all 0.3s ease;
}

.node-leaf:hover {
    filter: drop-shadow(0 3px 6px rgba(0,0,0,0.2));
}

/* Responsive */
@media (max-width: 768px) {
    .card-header .d-flex {
        flex-direction: column;
        gap: 1rem;
    }
    
    .legend .d-flex {
        flex-direction: column;
        gap: 0.5rem;
    }
}

/* Fullscreen styles */
#org-chart-container:fullscreen {
    padding: 20px;
    background: white;
}

#org-chart-container:fullscreen #org-chart {
    height: 100vh;
    overflow: auto;
}
</style>
@endsection