<?php

require_once 'vendor/autoload.php';

use App\Helpers\AssessmentHelper;
use App\Models\Question;
use App\Models\Assessment;
use App\Models\Answer;

// จำลองข้อมูลสำหรับทดสอบ
class MockAssessment
{
    public $answers;
    public $questionGroup;
    
    public function __construct()
    {
        $this->answers = collect();
        $this->questionGroup = new MockQuestionGroup();
    }
}

class MockQuestionGroup
{
    public $questions;
    
    public function __construct()
    {
        $this->questions = collect();
    }
}

// สร้างข้อมูลทดสอบ
function createTestData()
{
    // สร้างคำถามแม่ (ใช้ในการประเมิน)
    $parentQuestion = new Question();
    $parentQuestion->id = 1;
    $parentQuestion->question_text = "1. ข้อกำหนดทั่วไป";
    $parentQuestion->use_in_evaluation = true;
    $parentQuestion->logic_operator = 'AND';
    $parentQuestion->question_type = null; // คำถามแม่ไม่มี question_type
    
    // สร้างคำถามย่อย
    $child1 = new Question();
    $child1->id = 2;
    $child1->question_text = "1.1 มีเอกสาร";
    $child1->use_in_evaluation = true;
    $child1->question_type = 'yes_no';
    $child1->parent_id = 1;
    
    $child2 = new Question();
    $child2->id = 3;
    $child2->question_text = "1.2 มีการจัดเก็บ";
    $child2->use_in_evaluation = true;
    $child2->question_type = 'yes_no';
    $child2->parent_id = 1;
    
    $child3 = new Question();
    $child3->id = 4;
    $child3->question_text = "1.3 หมายเหตุ";
    $child3->use_in_evaluation = false; // ไม่ใช้ในการประเมิน
    $child3->question_type = 'text';
    $child3->parent_id = 1;
    
    $child4 = new Question();
    $child4->id = 5;
    $child4->question_text = "1.4 เลือกตัวเลือก";
    $child4->use_in_evaluation = true;
    $child4->question_type = 'multiple_choice'; // ไม่ใช่ yes_no
    $child4->parent_id = 1;
    
    // กำหนดความสัมพันธ์
    $parentQuestion->children = collect([$child1, $child2, $child3, $child4]);
    $child1->parent = $parentQuestion;
    $child2->parent = $parentQuestion;
    $child3->parent = $parentQuestion;
    $child4->parent = $parentQuestion;
    
    // สร้าง Assessment
    $assessment = new MockAssessment();
    $assessment->questionGroup->questions = collect([$parentQuestion]);
    
    return $assessment;
}

// ทดสอบกรณีต่างๆ
function testScenarios($assessment)
{
    echo "=== ทดสอบ Logic AND/OR เฉพาะคำถาม yes_no ที่ใช้ในการประเมิน ===\n\n";
    
    // ทดสอบกรณีที่ 1: คำถามย่อย yes_no ทั้งหมดตอบ "ใช่" (AND ควรเป็น true)
    echo "กรณีที่ 1: คำถามย่อย yes_no ทั้งหมดตอบ 'ใช่' (AND)\n";
    $assessment->answers = collect([
        createAnswer(2, 'ใช่'), // yes_no, ใช้ในการประเมิน
        createAnswer(3, 'ใช่'), // yes_no, ใช้ในการประเมิน
        createAnswer(4, 'มีหมายเหตุ'), // text, ไม่ใช้ในการประเมิน
        createAnswer(5, 'ตัวเลือก A') // multiple_choice, ใช้ในการประเมินแต่ไม่ใช่ yes_no
    ]);
    
    $results = AssessmentHelper::getEvaluationResults($assessment);
    $summary = AssessmentHelper::calculateSummary($assessment);
    
    echo "ผลลัพธ์: " . ($results[1]['result'] ? 'ผ่าน' : 'ไม่ผ่าน') . "\n";
    echo "จำนวนคำถามที่ประเมิน: {$summary['evaluated_questions']}\n";
    echo "จำนวนคำถามที่ผ่าน: {$summary['passed_questions']}\n";
    echo "เปอร์เซ็นต์: {$summary['percentage']}%\n";
    echo "คำถามที่นำมาคำนวณ: " . count($results[1]['children_results']) . " ข้อ (เฉพาะ yes_no ที่ใช้ในการประเมิน)\n\n";
    
    // ทดสอบกรณีที่ 2: คำถามย่อยหนึ่งตอบ "ไม่ใช่" (AND ควรเป็น false)
    echo "กรณีที่ 2: คำถามย่อยหนึ่งตอบ 'ไม่ใช่' (AND)\n";
    $assessment->answers = collect([
        createAnswer(2, 'ใช่'),
        createAnswer(3, 'ไม่ใช่')
    ]);
    
    $results = AssessmentHelper::getEvaluationResults($assessment);
    $summary = AssessmentHelper::calculateSummary($assessment);
    
    echo "ผลลัพธ์: " . ($results[1]['result'] ? 'ผ่าน' : 'ไม่ผ่าน') . "\n";
    echo "จำนวนคำถามที่ประเมิน: {$summary['evaluated_questions']}\n";
    echo "จำนวนคำถามที่ผ่าน: {$summary['passed_questions']}\n";
    echo "เปอร์เซ็นต์: {$summary['percentage']}%\n\n";
    
    // ทดสอบกรณีที่ 3: เปลี่ยนเป็น OR
    echo "กรณีที่ 3: เปลี่ยนเป็น OR\n";
    $assessment->questionGroup->questions->first()->logic_operator = 'OR';
    
    $results = AssessmentHelper::getEvaluationResults($assessment);
    $summary = AssessmentHelper::calculateSummary($assessment);
    
    echo "ผลลัพธ์: " . ($results[1]['result'] ? 'ผ่าน' : 'ไม่ผ่าน') . "\n";
    echo "จำนวนคำถามที่ประเมิน: {$summary['evaluated_questions']}\n";
    echo "จำนวนคำถามที่ผ่าน: {$summary['passed_questions']}\n";
    echo "เปอร์เซ็นต์: {$summary['percentage']}%\n\n";
    
    // ทดสอบกรณีที่ 4: ไม่มีคำตอบ
    echo "กรณีที่ 4: ไม่มีคำตอบ\n";
    $assessment->answers = collect([]);
    
    $results = AssessmentHelper::getEvaluationResults($assessment);
    $summary = AssessmentHelper::calculateSummary($assessment);
    
    echo "ผลลัพธ์: " . (isset($results[1]['result']) ? ($results[1]['result'] ? 'ผ่าน' : 'ไม่ผ่าน') : 'ไม่มีผล') . "\n";
    echo "จำนวนคำถามที่ประเมิน: {$summary['evaluated_questions']}\n";
    echo "จำนวนคำถามที่ผ่าน: {$summary['passed_questions']}\n";
    echo "เปอร์เซ็นต์: {$summary['percentage']}%\n\n";
    
    // ทดสอบกรณีที่ 5: ทดสอบคำถามเดี่ยวที่เป็น yes_no
    echo "กรณีที่ 5: ทดสอบคำถามเดี่ยวที่เป็น yes_no\n";
    $singleQuestion = new Question();
    $singleQuestion->id = 6;
    $singleQuestion->question_text = "2. คำถามเดี่ยว";
    $singleQuestion->use_in_evaluation = true;
    $singleQuestion->question_type = 'yes_no';
    $singleQuestion->parent_id = null;
    
    $assessment->questionGroup->questions->push($singleQuestion);
    $assessment->answers = collect([
        createAnswer(6, 'ใช่')
    ]);
    
    $results = AssessmentHelper::getEvaluationResults($assessment);
    $summary = AssessmentHelper::calculateSummary($assessment);
    
    echo "ผลลัพธ์: " . ($results[6]['result'] ? 'ผ่าน' : 'ไม่ผ่าน') . "\n";
    echo "จำนวนคำถามที่ประเมิน: {$summary['evaluated_questions']}\n";
    echo "จำนวนคำถามที่ผ่าน: {$summary['passed_questions']}\n";
    echo "เปอร์เซ็นต์: {$summary['percentage']}%\n\n";
}

function createAnswer($questionId, $answerText)
{
    $answer = new Answer();
    $answer->question_id = $questionId;
    $answer->answer_text = $answerText;
    return $answer;
}

// เรียกใช้ฟังก์ชันทดสอบ
$assessment = createTestData();
testScenarios($assessment);

echo "=== สรุปการเปลี่ยนแปลง ===\n";
echo "1. กรองเฉพาะคำถามที่ use_in_evaluation = true และ question_type = 'yes_no'\n";
echo "2. คำถามประเภทอื่นๆ (text, multiple_choice) จะไม่ถูกนำมาคำนวณ\n";
echo "3. คำถามที่ use_in_evaluation = false จะไม่ถูกนำมาคำนวณ\n";
echo "4. ระบบจะแสดงผลเฉพาะคำถาม yes_no ที่ใช้ในการประเมินเท่านั้น\n";
echo "5. การคำนวณเปอร์เซ็นต์จะนับเฉพาะคำถาม yes_no ที่ใช้ในการประเมิน\n"; 