<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Assessment;
use App\Models\Answer;
use App\Models\QuestionGroup;
use App\Models\Customer;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB; // Added DB facade import

class AssessmentController extends Controller
{
    public function create(Product $product)
    {
        if (Auth::user()->Customer_ID !== $product->user_id && Auth::user()->Level !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        $questionGroups = QuestionGroup::active()->with(['questions'])->get();
        return view('assessments.create', compact('product', 'questionGroups'));
    }

    public function store(Request $request)
    {
        // Debug: ตรวจสอบข้อมูลที่ส่งมา
        \Log::info('Assessment Store Request:', [
            'answers' => $request->answers,
            'session_pending_product' => session('pending_product')
        ]);

        $request->validate([
            'answers' => 'required|array',
            'answers.*.choice' => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text' => 'nullable|string',
        ]);

        // สร้างผลิตภัณฑ์จาก session ก่อน
        $pendingProduct = session('pending_product');
        $product = null;

        if ($pendingProduct) {
            $pendingProduct['user_id'] = Auth::user()->Customer_ID;
            $pendingProduct['CusIns_ID'] = Auth::user()->Customer_ID;
            $pendingProduct['CusUdp_ID'] = Auth::user()->Customer_ID;
            $product = Product::create($pendingProduct);
            $request->session()->forget('pending_product');
            
            \Log::info('Product created:', ['product_id' => $product->id]);
        } else {
            // ถ้าไม่มี session ให้ redirect กลับไปหน้าเพิ่มผลิตภัณฑ์
            return redirect()->route('products.create')
                ->with('error', 'กรุณากรอกข้อมูลผลิตภัณฑ์ก่อน');
        }

        try {
            $assessment = Assessment::create([
                'product_id' => $product->id,
                'question_group_id' => QuestionGroup::active()->first()->id,
                'user_id' => Auth::user()->Customer_ID,
                'CusIns_ID' => Auth::user()->Customer_ID,
                'CusUdp_ID' => Auth::user()->Customer_ID,
                'status' => 'submitted',
                'started_at' => now(),
                'submitted_at' => now(),
            ]);

            \Log::info('Assessment created:', ['assessment_id' => $assessment->id]);

            // ปิด foreign key checks ชั่วคราว
            \DB::statement('SET FOREIGN_KEY_CHECKS = 0');

            foreach ($request->answers as $questionId => $answerData) {
                \Log::info('Processing answer:', [
                    'question_id' => $questionId,
                    'question_id_type' => gettype($questionId),
                    'answer_data' => $answerData
                ]);

                // แปลง question_id เป็น integer
                $questionId = (int) $questionId;

                // ตรวจสอบว่าคำถามมีอยู่จริงหรือไม่
                $question = \App\Models\Question::find($questionId);
                if (!$question) {
                    \Log::error('Question not found:', ['question_id' => $questionId]);
                    continue;
                }

                \Log::info('Question found:', [
                    'question_id' => $questionId,
                    'question_text' => $question->question_text,
                    'question_type' => $question->question_type
                ]);

                // กำหนด answer_text ตามประเภทคำถาม
                $answerText = null;
                $comment = null;

                if (isset($answerData['choice'])) {
                    if ($answerData['choice'] === 'yes') {
                        $answerText = 'ใช่';
                    } elseif ($answerData['choice'] === 'no') {
                        $answerText = 'ไม่ใช่';
                    } else {
                        // สำหรับ multiple_choice
                        $answerText = $answerData['choice'];
                    }
                } elseif (isset($answerData['text'])) {
                    // สำหรับ text questions
                    $answerText = $answerData['text'];
                }

                // เก็บ details สำหรับ yes_no questions
                if (isset($answerData['details'])) {
                    $comment = $answerData['details'];
                }

                // บันทึกเฉพาะเมื่อมี answer_text
                if ($answerText) {
                    try {
                        $answer = Answer::create([
                            'assessment_id' => $assessment->id,
                            'question_id' => $questionId,
                            'answer_text' => $answerText,
                            'image_path' => null,
                            'comment' => $comment,
                            'CusIns_ID' => Auth::user()->Customer_ID,
                            'CusUdp_ID' => Auth::user()->Customer_ID,
                        ]);
                        
                        \Log::info('Answer created:', [
                            'answer_id' => $answer->id,
                            'question_id' => $questionId,
                            'answer_text' => $answerText
                        ]);
                    } catch (\Exception $e) {
                        \Log::error('Error creating answer:', [
                            'question_id' => $questionId,
                            'error' => $e->getMessage()
                        ]);
                        throw $e;
                    }
                }
            }

            // เปิด foreign key checks กลับ
            \DB::statement('SET FOREIGN_KEY_CHECKS = 1');

            return redirect()->route('products.report', $product)
                ->with('success', 'บันทึกการประเมินเรียบร้อยแล้ว');

        } catch (\Exception $e) {
            // เปิด foreign key checks กลับในกรณี error
            \DB::statement('SET FOREIGN_KEY_CHECKS = 1');
            
            \Log::error('Assessment store error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'เกิดข้อผิดพลาดในการบันทึกการประเมิน: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * บันทึกฉบับร่างการประเมิน
     */
    public function storeDraft(Request $request)
    {
        // Debug: ตรวจสอบข้อมูลที่ส่งมา
        \Log::info('Assessment Draft Store Request:', [
            'answers' => $request->answers,
            'session_pending_product' => session('pending_product')
        ]);

        $request->validate([
            'answers' => 'nullable|array',
            'answers.*.choice' => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text' => 'nullable|string',
        ]);

        // สร้างผลิตภัณฑ์จาก session ก่อน
        $pendingProduct = session('pending_product');
        $product = null;

        if ($pendingProduct) {
            $pendingProduct['user_id'] = Auth::user()->Customer_ID;
            $pendingProduct['CusIns_ID'] = Auth::user()->Customer_ID;
            $pendingProduct['CusUdp_ID'] = Auth::user()->Customer_ID;
            $product = Product::create($pendingProduct);
            $request->session()->forget('pending_product');
            
            \Log::info('Product created for draft:', ['product_id' => $product->id]);
        } else {
            // ถ้าไม่มี session ให้ redirect กลับไปหน้าเพิ่มผลิตภัณฑ์
            return redirect()->route('products.create')
                ->with('error', 'กรุณากรอกข้อมูลผลิตภัณฑ์ก่อน');
        }

        try {
            // ตรวจสอบว่ามี assessment draft อยู่แล้วหรือไม่
            $existingAssessment = Assessment::where('product_id', $product->id)
                ->where('status', 'draft')
                ->first();

            if ($existingAssessment) {
                // อัปเดต assessment ที่มีอยู่
                $assessment = $existingAssessment;
                $assessment->update([
                    'CusUdp_ID' => Auth::user()->Customer_ID,
                    'DtmUpd' => now(),
                ]);
                
                // ลบคำตอบเก่า
                $assessment->answers()->delete();
            } else {
                // สร้าง assessment ใหม่
                $assessment = Assessment::create([
                    'product_id' => $product->id,
                    'question_group_id' => QuestionGroup::active()->first()->id,
                    'user_id' => Auth::user()->Customer_ID,
                    'CusIns_ID' => Auth::user()->Customer_ID,
                    'CusUdp_ID' => Auth::user()->Customer_ID,
                    'status' => 'draft',
                    'started_at' => now(),
                    'submitted_at' => null,
                ]);
            }

            \Log::info('Assessment draft created/updated:', ['assessment_id' => $assessment->id]);

            // ปิด foreign key checks ชั่วคราว
            \DB::statement('SET FOREIGN_KEY_CHECKS = 0');

            // บันทึกคำตอบ (ถ้ามี)
            if ($request->has('answers') && is_array($request->answers)) {
                foreach ($request->answers as $questionId => $answerData) {
                    \Log::info('Processing draft answer:', [
                        'question_id' => $questionId,
                        'answer_data' => $answerData
                    ]);

                    $questionId = (int) $questionId;
                    $question = \App\Models\Question::find($questionId);
                    
                    if (!$question) {
                        \Log::error('Question not found for draft:', ['question_id' => $questionId]);
                        continue;
                    }

                    $answerText = null;
                    $comment = null;

                    if (isset($answerData['choice'])) {
                        if ($answerData['choice'] === 'yes') {
                            $answerText = 'ใช่';
                        } elseif ($answerData['choice'] === 'no') {
                            $answerText = 'ไม่ใช่';
                        } else {
                            $answerText = $answerData['choice'];
                        }
                    } elseif (isset($answerData['text'])) {
                        $answerText = $answerData['text'];
                    }

                    if (isset($answerData['details'])) {
                        $comment = $answerData['details'];
                    }

                    if ($answerText) {
                        try {
                            $answer = Answer::create([
                                'assessment_id' => $assessment->id,
                                'question_id' => $questionId,
                                'answer_text' => $answerText,
                                'image_path' => null,
                                'comment' => $comment,
                                'CusIns_ID' => Auth::user()->Customer_ID,
                                'CusUdp_ID' => Auth::user()->Customer_ID,
                            ]);
                            
                            \Log::info('Draft answer created:', [
                                'answer_id' => $answer->id,
                                'question_id' => $questionId,
                                'answer_text' => $answerText
                            ]);
                        } catch (\Exception $e) {
                            \Log::error('Error creating draft answer:', [
                                'question_id' => $questionId,
                                'error' => $e->getMessage()
                            ]);
                            throw $e;
                        }
                    }
                }
            }

            // เปิด foreign key checks กลับ
            \DB::statement('SET FOREIGN_KEY_CHECKS = 1');

            return redirect()->route('products.index')
                ->with('success', 'บันทึกฉบับร่างเรียบร้อยแล้ว');

        } catch (\Exception $e) {
            // เปิด foreign key checks กลับในกรณี error
            \DB::statement('SET FOREIGN_KEY_CHECKS = 1');
            
            \Log::error('Assessment draft store error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'เกิดข้อผิดพลาดในการบันทึกฉบับร่าง: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * แสดงฟอร์มแก้ไขการประเมิน
     */
    public function edit(Assessment $assessment)
    {
        // ตรวจสอบสิทธิ์
        if (Auth::user()->Customer_ID !== $assessment->user_id && Auth::user()->Level !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        // ดึงข้อมูลคำถาม
        $questionGroups = QuestionGroup::active()->with(['questions'])->get();
        
        return view('assessments.edit', compact('assessment', 'questionGroups'));
    }

    /**
     * อัปเดตการประเมิน
     */
    public function update(Request $request, Assessment $assessment)
    {
        // ตรวจสอบสิทธิ์
        if (Auth::user()->Customer_ID !== $assessment->user_id && Auth::user()->Level !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        $request->validate([
            'answers' => 'nullable|array',
            'answers.*.choice' => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text' => 'nullable|string',
        ]);

        try {
            // อัปเดตสถานะเป็น submitted
            $assessment->update([
                'status' => 'submitted',
                'submitted_at' => now(),
                'CusUdp_ID' => Auth::user()->Customer_ID,
            ]);

            // ปิด foreign key checks ชั่วคราว
            \DB::statement('SET FOREIGN_KEY_CHECKS = 0');

            // ลบคำตอบเก่า
            $assessment->answers()->delete();

            // บันทึกคำตอบใหม่
            if ($request->has('answers') && is_array($request->answers)) {
                foreach ($request->answers as $questionId => $answerData) {
                    $questionId = (int) $questionId;
                    $question = \App\Models\Question::find($questionId);
                    
                    if (!$question) {
                        continue;
                    }

                    $answerText = null;
                    $comment = null;

                    if (isset($answerData['choice'])) {
                        if ($answerData['choice'] === 'yes') {
                            $answerText = 'ใช่';
                        } elseif ($answerData['choice'] === 'no') {
                            $answerText = 'ไม่ใช่';
                        } else {
                            $answerText = $answerData['choice'];
                        }
                    } elseif (isset($answerData['text'])) {
                        $answerText = $answerData['text'];
                    }

                    if (isset($answerData['details'])) {
                        $comment = $answerData['details'];
                    }

                    if ($answerText) {
                        Answer::create([
                            'assessment_id' => $assessment->id,
                            'question_id' => $questionId,
                            'answer_text' => $answerText,
                            'image_path' => null,
                            'comment' => $comment,
                            'CusIns_ID' => Auth::user()->Customer_ID,
                            'CusUdp_ID' => Auth::user()->Customer_ID,
                        ]);
                    }
                }
            }

            // เปิด foreign key checks กลับ
            \DB::statement('SET FOREIGN_KEY_CHECKS = 1');

            return redirect()->route('products.report', $assessment->product)
                ->with('success', 'ส่งแบบประเมินเรียบร้อยแล้ว');

        } catch (\Exception $e) {
            // เปิด foreign key checks กลับในกรณี error
            \DB::statement('SET FOREIGN_KEY_CHECKS = 1');
            
            return redirect()->back()
                ->with('error', 'เกิดข้อผิดพลาดในการส่งแบบประเมิน: ' . $e->getMessage())
                ->withInput();
        }
    }
} 

