<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Question extends Model
{
    protected $fillable = [
        'question_text',
        'description',
        'question_group_id',
        'parent_id',
        'use_in_evaluation',
        'logic_operator',
        'keyword_tag',
        'sort_order',
        'question_type',
        'choices',
        'CusIns_ID',
        'CusUdp_ID'
    ];

    protected $casts = [
        'use_in_evaluation' => 'boolean',
        'choices' => 'array'
    ];

    // กำหนดชื่อ timestamp columns
    const CREATED_AT = 'DtmIns';
    const UPDATED_AT = 'DtmUpd';

    /**
     * ความสัมพันธ์: Question อยู่ใน QuestionGroup
     */
    public function questionGroup()
    {
        return $this->belongsTo(QuestionGroup::class);
    }

    /**
     * ความสัมพันธ์กับผู้สร้าง (Customer)
     */
    public function creator()
    {
        return $this->belongsTo(Customer::class, 'CusIns_ID', 'Customer_ID');
    }

    /**
     * ความสัมพันธ์กับผู้แก้ไข (Customer)
     */
    public function updater()
    {
        return $this->belongsTo(Customer::class, 'CusUdp_ID', 'Customer_ID');
    }

    /**
     * ความสัมพันธ์: คำถามแม่
     */
    public function parent()
    {
        return $this->belongsTo(Question::class, 'parent_id');
    }

    /**
     * ความสัมพันธ์: คำถามย่อย
     */
    public function children()
    {
        return $this->hasMany(Question::class, 'parent_id')->orderBy('sort_order');
    }

    /**
     * ตรวจสอบว่ามีคำถามย่อยหรือไม่
     */
    public function hasChildren()
    {
        return $this->children()->count() > 0;
    }

    /**
     * ตรวจสอบว่าเป็นคำถามย่อยหรือไม่
     */
    public function isChild()
    {
        return !is_null($this->parent_id);
    }

    /**
     * Accessor สำหรับ logic operator label
     */
    public function getLogicOperatorLabelAttribute()
    {
        return $this->logic_operator ?: 'AND';
    }

    /**
     * Accessor สำหรับ question type label
     */
    public function getQuestionTypeLabelAttribute()
    {
        $types = [
            'yes_no' => 'ใช่/ไม่ใช่',
            'multiple_choice' => 'ตัวเลือก',
            'text' => 'ข้อความ',
            null => 'ไม่มี'
        ];
        
        return $types[$this->question_type] ?? 'ไม่ระบุ';
    }
}
