<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Product;
use App\Models\Assessment;
use App\Models\Answer;
use App\Models\Question;
use App\Models\QuestionGroup;
use App\Models\Customer;
use App\Models\Company;
use App\Models\Agency;
use App\Models\AgencyCustomer;

class ProductController extends Controller
{
    /**
     * Display a listing of products from the same company.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $filter = $request->get('filter');
        $selectedAgencyId = $request->get('agency_id') ?? session('selected_agency_id');

        $userAgencies = collect();
        $selectedAgency = null;

        if (strtolower($user->Level) === 'consult') {
            // รายชื่อบริษัทที่ consult เข้าถึงได้
            $userAgencies = $this->getConsultUserAgencies($user);

            if (!$selectedAgencyId && $userAgencies->isNotEmpty()) {
                $selectedAgencyId = $userAgencies->first()->Agency_ID;
            }

            session(['selected_agency_id' => $selectedAgencyId]);
            $selectedAgency = $userAgencies->where('Agency_ID', $selectedAgencyId)->first();

            if ($selectedAgency) {
                if ($filter === 'my') {
                    $products = Product::where('Customer_ID', $user->Customer_ID)
                        ->where('Agency_ID', $selectedAgencyId)
                        ->with('user', 'agency')
                        ->orderBy('DtmIns', 'asc')
                        ->get();
                } else {
                    $products = Product::where('Agency_ID', $selectedAgencyId)
                        ->with('user', 'agency')
                        ->orderBy('DtmIns', 'asc')
                        ->get();
                }
            } else {
                $products = collect();
            }
        } else {
            // user ปกติ
            $selectedAgencyId = $user->Agency_ID;
            session(['selected_agency_id' => $selectedAgencyId]);

            if ($filter !== 'my') {
                $products = Product::where('Agency_ID', $selectedAgencyId)
                    ->with('user', 'agency')
                    ->orderBy('DtmIns', 'asc')
                    ->get();
            } else {
                $products = Product::where('Customer_ID', $user->Customer_ID)
                    ->with('user', 'agency')
                    ->orderBy('DtmIns', 'asc')
                    ->get();
            }
        }

        return view('products.index', compact('products', 'userAgencies', 'selectedAgency', 'selectedAgencyId'));
    }

    /**
     * ดึงรายการบริษัทที่ consult user สามารถเข้าถึงได้
     */
    private function getConsultUserAgencies($user)
    {
        $agencyIds = AgencyCustomer::where('Customer_ID', $user->Customer_ID)
            ->pluck('Agency_ID')
            ->toArray();

        if ($user->Agency_ID && !in_array($user->Agency_ID, $agencyIds)) {
            $agencyIds[] = $user->Agency_ID;
        }

        return Agency::whereIn('Agency_ID', $agencyIds)->get();
    }

    /**
     * Show the form for creating a new product.
     */
    public function create(Request $request)
    {
        $selectedAgencyId = $request->get('agency_id') ?? session('selected_agency_id');
        $validated = session('pending_product');

        if ($validated) {
            $questionGroups = QuestionGroup::active()->with(['questions'])->get();
            return view('products.create', compact('validated', 'questionGroups', 'selectedAgencyId'));
        }

        return view('products.create', compact('selectedAgencyId'));
    }

    /**
     * Start new product creation (clear session)
     */
    public function createNew(Request $request)
    {
        $request->session()->forget('pending_product');
        $request->session()->forget('pending_assessment');

        $selectedAgencyId = $request->get('agency_id') ?? session('selected_agency_id');

        return view('products.create', compact('selectedAgencyId'));
    }

    /**
     * Store a newly created product in storage (เก็บไว้ใน session รอกดประเมิน).
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'               => 'required|string|max:255',
            'model'              => 'nullable|string|max:255',
            'log_receive_method' => 'required|in:syslog_udp,syslog_tcp,syslog_both,other',
            'product_type'       => 'required|in:hw_sw_log_server,sw_log_server',
            'description'        => 'nullable|string',
            'image'              => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        try {
            $user = Auth::user();

            // อัปโหลดรูป
            if ($request->hasFile('image')) {
                $path = $request->file('image')->store('products_images', 'public');
                $validated['image'] = $path;
            }

            // หาบริษัทที่จะผูก
            $selectedAgencyId = $request->input('selected_agency_id') ?? session('selected_agency_id') ?? $user->Agency_ID;
            session(['selected_agency_id' => $selectedAgencyId]);

            // เติมฟิลด์ตามสคีมา
            $customerId = $user->Customer_ID;
            $now = now();
            $validated['Customer_ID'] = $customerId;
            $validated['CusIns_ID']   = $customerId;
            $validated['CusUdp_ID']   = $customerId;
            $validated['DtmIns']      = $now;
            $validated['DtmUpd']      = $now;
            $validated['Agency_ID']   = $selectedAgencyId;

            // เก็บลง session ก่อน (รอทำแบบประเมิน)
            $request->session()->put('pending_product', $validated);

            $questionGroups = QuestionGroup::active()->with(['questions'])->get();
            return view('products.create', compact('validated', 'questionGroups', 'selectedAgencyId'));

        } catch (\Exception $e) {
            if (isset($validated['image']) && Storage::disk('public')->exists($validated['image'])) {
                Storage::disk('public')->delete($validated['image']);
            }
            return back()->withInput()->with('error', 'เกิดข้อผิดพลาดในการเพิ่มผลิตภัณฑ์ กรุณาลองใหม่อีกครั้ง');
        }
    }

    public function destroy(Product $product)
    {
        $user = Auth::user();
        $hasDeleteAccess = false;

        if ($user->Customer_ID === $product->Customer_ID) {
            $hasDeleteAccess = true;
        } elseif (in_array(strtolower($user->Level), ['admin','superadmin'])) {
            $hasDeleteAccess = true;
        } elseif ($user->Agency_ID && $product->user && $user->Agency_ID === $product->user->Agency_ID) {
            $hasDeleteAccess = true;
        } elseif (strtolower($user->Level) === 'consult' && $product->Agency_ID) {
            $userAgencies = $this->getConsultUserAgencies($user);
            if ($userAgencies->where('Agency_ID', $product->Agency_ID)->isNotEmpty()) {
                $hasDeleteAccess = true;
            }
        }

        if (!$hasDeleteAccess) {
            return redirect()->route('products.index')->with('error', 'คุณไม่มีสิทธิในการลบผลิตภัณฑ์นี้');
        }

        try {
            $product->delete();
            return redirect()->route('products.index')->with('success', 'ผลิตภัณฑ์ถูกลบอย่างสำเร็จ');
        } catch (\Exception $e) {
            return redirect()->route('products.index')->with('error', 'เกิดข้อผิดพลาดในการลบผลิตภัณฑ์ กรุณาลองใหม่อีกครั้ง');
        }
    }

    public function report(Product $product)
    {
        $user = Auth::user();
        $hasAccess = false;

        if ($user->Customer_ID === $product->Customer_ID) {
            $hasAccess = true;
        } elseif (in_array(strtolower($user->Level), ['admin','superadmin'])) {
            $hasAccess = true;
        } elseif ($user->Agency_ID && $product->user && $user->Agency_ID === $product->user->Agency_ID) {
            $hasAccess = true;
        } elseif (strtolower($user->Level) === 'consult' && $product->Agency_ID) {
            $userAgencies = $this->getConsultUserAgencies($user);
            if ($userAgencies->where('Agency_ID', $product->Agency_ID)->isNotEmpty()) {
                $hasAccess = true;
            }
        }

        if (!$hasAccess) {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        $assessment = $product->assessments()->with(['answers.question'])->latest()->first();

        if (!$assessment) {
            return view('products.report', compact('product'))->with('assessment', null);
        }

        $answeredQuestions = collect();
        foreach ($assessment->answers as $answer) {
            if ($answer->question) {
                $answer->question->answer = $answer;
                $answeredQuestions->push($answer->question);
            }
        }

        $answeredQuestions = $answeredQuestions->sortBy(fn($q) => $q->question_number_sort_key ?? $q->sort_order ?? 0);

        return view('products.report', compact('product', 'assessment', 'answeredQuestions'));
    }
}
