<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Assessment;
use App\Models\Answer;
use App\Models\QuestionGroup;
use App\Models\Customer;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB; // Added DB facade import

class AssessmentController extends Controller
{
    public function create(Product $product)
    {
        if (Auth::user()->Customer_ID !== $product->user_id && strtolower(Auth::user()->Level) !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        $questionGroups = QuestionGroup::active()->with(['questions'])->get();
        return view('assessments.create', compact('product', 'questionGroups'));
    }

    public function store(Request $request)
    {
        // Debug: ตรวจสอบข้อมูลที่ส่งมา
        Log::info('Assessment Store Request:', [
            'answers' => $request->answers,
            'session_pending_product' => session('pending_product')
        ]);

        $request->validate([
            'answers' => 'required|array',
            'answers.*.choice' => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text' => 'nullable|string',
        ]);

        // สร้างผลิตภัณฑ์จาก session ก่อน
        $pendingProduct = session('pending_product');
        $product = null;

        if ($pendingProduct) {
            $user = Auth::user();
            $pendingProduct['user_id'] = $user->Customer_ID;
            $pendingProduct['CusIns_ID'] = $user->Customer_ID;
            $pendingProduct['CusUdp_ID'] = $user->Customer_ID;
            
            // สำหรับ consult level ต้องจัดการ Agency_ID ของผลิตภัณฑ์ให้ตรงกับบริษัทที่เลือก
            if (strtolower($user->Level) === 'consult') {
                $selectedAgencyId = $request->get('selected_agency_id') ?? session('selected_agency_id');
                if ($selectedAgencyId) {
                    // ตรวจสอบว่า consult user มีสิทธิ์เข้าถึงบริษัทนี้หรือไม่
                    $userAgencies = $this->getConsultUserAgencies($user);
                    if ($userAgencies->where('Agency_ID', $selectedAgencyId)->isNotEmpty()) {
                        $pendingProduct['Agency_ID'] = $selectedAgencyId;
                    }
                }
            } else {
                // สำหรับ level อื่นๆ ใช้ Agency_ID ของ user
                $pendingProduct['Agency_ID'] = $user->Agency_ID;
            }
            
            $product = Product::create($pendingProduct);
            $request->session()->forget('pending_product');
            $request->session()->forget('selected_agency_id'); // ลบ session หลังใช้แล้ว
            
            Log::info('Product created:', ['product_id' => $product->id, 'Agency_ID' => $pendingProduct['Agency_ID'] ?? 'null']);
        } else {
            // ถ้าไม่มี session ให้ redirect กลับไปหน้าเพิ่มผลิตภัณฑ์
            return redirect()->route('products.index')
                ->with('error', 'กรุณากรอกข้อมูลผลิตภัณฑ์ก่อน');
        }

        try {
            $assessment = Assessment::create([
                'product_id' => $product->id,
                'question_group_id' => QuestionGroup::active()->first()->id,
                'user_id' => Auth::user()->Customer_ID,
                'CusIns_ID' => Auth::user()->Customer_ID,
                'CusUdp_ID' => Auth::user()->Customer_ID,
                'status' => 'submitted',
                'started_at' => now(),
                'submitted_at' => now(),
            ]);

            Log::info('Assessment created:', ['assessment_id' => $assessment->id]);

            // ปิด foreign key checks ชั่วคราว
            DB::statement('SET FOREIGN_KEY_CHECKS = 0');

            foreach ($request->answers as $questionId => $answerData) {
                Log::info('Processing answer:', [
                    'question_id' => $questionId,
                    'question_id_type' => gettype($questionId),
                    'answer_data' => $answerData
                ]);

                // แปลง question_id เป็น integer
                $questionId = (int) $questionId;

                // ตรวจสอบว่าคำถามมีอยู่จริงหรือไม่
                $question = \App\Models\Question::find($questionId);
                if (!$question) {
                    Log::error('Question not found:', ['question_id' => $questionId]);
                    continue;
                }

                Log::info('Question found:', [
                    'question_id' => $questionId,
                    'question_text' => $question->question_text,
                    'question_type' => $question->question_type,
                    'use_in_evaluation' => $question->use_in_evaluation
                ]);

                // กำหนด answer_text ตามประเภทคำถาม
                $answerText = null;
                $comment = null;

                if (isset($answerData['choice'])) {
                    if ($answerData['choice'] === 'yes') {
                        $answerText = 'ใช่';
                    } elseif ($answerData['choice'] === 'no') {
                        $answerText = 'ไม่ใช่';
                    } else {
                        // สำหรับ multiple_choice
                        $answerText = $answerData['choice'];
                    }
                } elseif (isset($answerData['text'])) {
                    // สำหรับ text questions
                    $answerText = $answerData['text'];
                }

                // เก็บ details สำหรับ yes_no questions
                if (isset($answerData['details'])) {
                    $comment = $answerData['details'];
                }

                // บันทึกคำตอบสำหรับคำถามที่มี question_type ไม่เป็น null
                if ($answerText && $question->question_type !== null) {
                    try {
                        $answer = Answer::create([
                            'assessment_id' => $assessment->id,
                            'question_id' => $questionId,
                            'answer_text' => $answerText,
                            'image_path' => null,
                            'comment' => $comment,
                            'CusIns_ID' => Auth::user()->Customer_ID,
                            'CusUdp_ID' => Auth::user()->Customer_ID,
                        ]);
                        
                        Log::info('Answer created:', [
                            'answer_id' => $answer->id,
                            'question_id' => $questionId,
                            'answer_text' => $answerText
                        ]);
                    } catch (\Exception $e) {
                        Log::error('Error creating answer:', [
                            'question_id' => $questionId,
                            'error' => $e->getMessage()
                        ]);
                        throw $e;
                    }
                }
            }

            // ประมวลผล parent questions ที่มี question_type = null และ use_in_evaluation = 1
            $parentQuestions = \App\Models\Question::where('question_type', null)
                ->where('use_in_evaluation', 1)
                ->whereHas('questionGroup', function($query) {
                    $query->where('is_active', true);
                })
                ->get();

            // ฟังก์ชันสำหรับคำนวณผลลัพธ์ของ parent question
            $calculateParentResult = function($parentQuestionId) use ($assessment, &$calculateParentResult) {
                $childQuestions = \App\Models\Question::where('parent_id', $parentQuestionId)
                    ->where('use_in_evaluation', 1)
                    ->get();

                if ($childQuestions->count() == 0) {
                    return false;
                }

                $childAnswers = [];
                foreach ($childQuestions as $childQuestion) {
                    if ($childQuestion->question_type === 'yes_no') {
                        // สำหรับ yes_no questions
                        $childAnswer = $assessment->answers()->where('question_id', $childQuestion->id)->first();
                        if ($childAnswer) {
                            $childAnswers[] = $childAnswer->answer_text === 'ใช่';
                        } else {
                            $childAnswers[] = false;
                        }
                    } elseif ($childQuestion->question_type === null) {
                        // สำหรับ nested parent questions - เรียก recursive
                        $childAnswers[] = $calculateParentResult($childQuestion->id);
                    }
                }

                // คำนวณผลลัพธ์ตาม logic operator
                $parentQuestion = \App\Models\Question::find($parentQuestionId);
                if ($parentQuestion->logic_operator === 'AND') {
                    return !in_array(false, $childAnswers);
                } elseif ($parentQuestion->logic_operator === 'OR') {
                    return in_array(true, $childAnswers);
                }
                
                return false;
            };

            foreach ($parentQuestions as $parentQuestion) {
                $result = $calculateParentResult($parentQuestion->id);

                // บันทึกผลลัพธ์ของ parent question
                try {
                    $parentAnswer = Answer::create([
                        'assessment_id' => $assessment->id,
                        'question_id' => $parentQuestion->id,
                        'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                        'image_path' => null,
                        'comment' => null,
                        'CusIns_ID' => Auth::user()->Customer_ID,
                        'CusUdp_ID' => Auth::user()->Customer_ID,
                    ]);
                    
                    Log::info('Parent answer created:', [
                        'answer_id' => $parentAnswer->id,
                        'question_id' => $parentQuestion->id,
                        'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                        'logic_operator' => $parentQuestion->logic_operator
                    ]);
                } catch (\Exception $e) {
                    Log::error('Error creating parent answer:', [
                        'question_id' => $parentQuestion->id,
                        'error' => $e->getMessage()
                    ]);
                    throw $e;
                }
            }

            // เปิด foreign key checks กลับ
            DB::statement('SET FOREIGN_KEY_CHECKS = 1');

            return redirect()->route('products.report', $product)
                ->with('success', 'บันทึกการประเมินเรียบร้อยแล้ว');

        } catch (\Exception $e) {
            // เปิด foreign key checks กลับในกรณี error
            DB::statement('SET FOREIGN_KEY_CHECKS = 1');
            
            Log::error('Assessment store error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'เกิดข้อผิดพลาดในการบันทึกการประเมิน: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
 * ส่งแบบประเมินจากฉบับร่าง
 */
public function submitDraft(Request $request)
{
    // ตรวจสอบว่ามี product_id หรือไม่
    if (!$request->has('product_id')) {
        return redirect()->back()
            ->with('error', 'ไม่พบข้อมูลผลิตภัณฑ์');
    }

    $productId = $request->product_id;
    
    // หา assessment ที่เป็น draft สำหรับผลิตภัณฑ์นี้
    $assessment = Assessment::where('product_id', $productId)
        ->where('status', 'draft')
        ->where('user_id', Auth::user()->Customer_ID)
        ->first();

    if (!$assessment) {
        return redirect()->back()
            ->with('error', 'ไม่พบฉบับร่างหรือไม่มีสิทธิ์เข้าถึง');
    }

    // ตรวจสอบสิทธิ์
    if (Auth::user()->Customer_ID !== $assessment->user_id && strtolower(Auth::user()->Level) !== 'admin') {
        abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
    }

    // ตรวจสอบว่าเป็น draft หรือไม่
    if ($assessment->status !== 'draft') {
        return redirect()->back()
            ->with('error', 'ไม่สามารถส่งแบบประเมินได้');
    }

    // Validation
    $request->validate([
        'answers' => 'required|array',
        'answers.*.choice' => 'nullable|string',
        'answers.*.details' => 'nullable|string',
        'answers.*.text' => 'nullable|string',
    ]);

    try {
        DB::beginTransaction();

        // อัปเดตสถานะเป็น submitted
        $assessment->update([
            'status' => 'submitted',
            'submitted_at' => now(),
            'CusUdp_ID' => Auth::user()->Customer_ID,
            'DtmUpd' => now(),
        ]);

        // ลบคำตอบเก่าและบันทึกใหม่
        $assessment->answers()->delete();
        
        // บันทึกคำตอบใหม่ (ใช้ logic เดียวกับ store method)
        foreach ($request->answers as $questionId => $answerData) {
            $questionId = (int) $questionId;
            $question = \App\Models\Question::find($questionId);
            
            if (!$question) {
                continue;
            }

            $answerText = null;
            $comment = null;

            if (isset($answerData['choice'])) {
                if ($answerData['choice'] === 'yes') {
                    $answerText = 'ใช่';
                } elseif ($answerData['choice'] === 'no') {
                    $answerText = 'ไม่ใช่';
                } else {
                    $answerText = $answerData['choice'];
                }
            } elseif (isset($answerData['text'])) {
                $answerText = $answerData['text'];
            }

            if (isset($answerData['details'])) {
                $comment = $answerData['details'];
            }

            // บันทึกคำตอบสำหรับคำถามที่มี question_type ไม่เป็น null
            if ($answerText && $question->question_type !== null) {
                Answer::create([
                    'assessment_id' => $assessment->id,
                    'question_id' => $questionId,
                    'answer_text' => $answerText,
                    'image_path' => null,
                    'comment' => $comment,
                    'CusIns_ID' => Auth::user()->Customer_ID,
                    'CusUdp_ID' => Auth::user()->Customer_ID,
                ]);
            }
        }

        // ประมวลผล parent questions (ใช้ logic เดียวกับ store method)
        $parentQuestions = \App\Models\Question::where('question_type', null)
            ->where('use_in_evaluation', 1)
            ->whereHas('questionGroup', function($query) {
                $query->where('is_active', true);
            })
            ->get();

        // ฟังก์ชันสำหรับคำนวณผลลัพธ์ของ parent question
        $calculateParentResult = function($parentQuestionId) use ($assessment, &$calculateParentResult) {
            $childQuestions = \App\Models\Question::where('parent_id', $parentQuestionId)
                ->where('use_in_evaluation', 1)
                ->get();

            if ($childQuestions->count() == 0) {
                return false;
            }

            $childAnswers = [];
            foreach ($childQuestions as $childQuestion) {
                if ($childQuestion->question_type === 'yes_no') {
                    $childAnswer = $assessment->answers()->where('question_id', $childQuestion->id)->first();
                    if ($childAnswer) {
                        $childAnswers[] = $childAnswer->answer_text === 'ใช่';
                    } else {
                        $childAnswers[] = false;
                    }
                } elseif ($childQuestion->question_type === null) {
                    $childAnswers[] = $calculateParentResult($childQuestion->id);
                }
            }

            $parentQuestion = \App\Models\Question::find($parentQuestionId);
            if ($parentQuestion->logic_operator === 'AND') {
                return !in_array(false, $childAnswers);
            } elseif ($parentQuestion->logic_operator === 'OR') {
                return in_array(true, $childAnswers);
            }
            
            return false;
        };

        foreach ($parentQuestions as $parentQuestion) {
            $result = $calculateParentResult($parentQuestion->id);

            Answer::create([
                'assessment_id' => $assessment->id,
                'question_id' => $parentQuestion->id,
                'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                'image_path' => null,
                'comment' => null,
                'CusIns_ID' => Auth::user()->Customer_ID,
                'CusUdp_ID' => Auth::user()->Customer_ID,
            ]);
        }

        DB::commit();

        // ลบ session pending_product ถ้ามี
        session()->forget('pending_product');

        // ตรวจสอบว่า product มีอยู่จริงหรือไม่
        $product = Product::find($productId);
        if (!$product) {
            // สำหรับ consult level ให้ redirect กลับไปยังบริษัทที่เลือก
            $user = Auth::user();
            if (strtolower($user->Level) === 'consult') {
                $selectedAgencyId = $request->get('selected_agency_id');
                if ($selectedAgencyId) {
                    return redirect()->route('products.index', ['agency_id' => $selectedAgencyId])
                        ->with('error', 'ไม่พบข้อมูลผลิตภัณฑ์');
                }
            }
            return redirect()->route('products.index')
                ->with('error', 'ไม่พบข้อมูลผลิตภัณฑ์');
        }

        // สำหรับ consult level ให้ redirect ไปยังรายงานพร้อมพารามิเตอร์ agency_id
        $user = Auth::user();
        if (strtolower($user->Level) === 'consult') {
            $selectedAgencyId = $request->get('selected_agency_id');
            if ($selectedAgencyId) {
                return redirect()->route('products.report', $product)
                    ->with('success', 'ส่งแบบประเมินเรียบร้อยแล้ว');
            }
        }

        // Redirect ไปหน้ารายงาน
        return redirect()->route('products.report', $product)
            ->with('success', 'ส่งแบบประเมินเรียบร้อยแล้ว');

    } catch (\Exception $e) {
        DB::rollback();
        
        Log::error('Error submitting draft assessment: ' . $e->getMessage());
        
        return redirect()->back()
            ->with('error', 'เกิดข้อผิดพลาดในการส่งแบบประเมิน: ' . $e->getMessage());
    }
}
    /**
     * บันทึกฉบับร่างการประเมิน
     */
    public function storeDraft(Request $request)
    {
        // Debug: ตรวจสอบข้อมูลที่ส่งมา
        Log::info('Assessment Draft Store Request:', [
            'answers' => $request->answers,
            'session_pending_product' => session('pending_product')
        ]);

        $request->validate([
            'answers' => 'nullable|array',
            'answers.*.choice' => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text' => 'nullable|string',
        ]);

        // ตรวจสอบว่ามี assessment draft อยู่แล้วหรือไม่
        $assessment = null;
        $product = null;

        // ถ้ามี session pending_product ให้สร้างผลิตภัณฑ์ใหม่
        if (session('pending_product')) {
            $user = Auth::user();
            $pendingProduct = session('pending_product');
            $pendingProduct['user_id'] = $user->Customer_ID;
            $pendingProduct['CusIns_ID'] = $user->Customer_ID;
            $pendingProduct['CusUdp_ID'] = $user->Customer_ID;
            
            // สำหรับ consult level ต้องจัดการ Agency_ID ของผลิตภัณฑ์ให้ตรงกับบริษัทที่เลือก
            if (strtolower($user->Level) === 'consult') {
                $selectedAgencyId = $request->get('selected_agency_id') ?? session('selected_agency_id');
                if ($selectedAgencyId) {
                    // ตรวจสอบว่า consult user มีสิทธิ์เข้าถึงบริษัทนี้หรือไม่
                    $userAgencies = $this->getConsultUserAgencies($user);
                    if ($userAgencies->where('Agency_ID', $selectedAgencyId)->isNotEmpty()) {
                        $pendingProduct['Agency_ID'] = $selectedAgencyId;
                    }
                }
            } else {
                // สำหรับ level อื่นๆ ใช้ Agency_ID ของ user
                $pendingProduct['Agency_ID'] = $user->Agency_ID;
            }
            
            $product = Product::create($pendingProduct);
            $request->session()->forget('pending_product');
            
            Log::info('Product created for draft:', ['product_id' => $product->id, 'Agency_ID' => $pendingProduct['Agency_ID'] ?? 'null']);
        } else {
            // ถ้าไม่มี session ให้ตรวจสอบว่ามี assessment draft อยู่แล้วหรือไม่
            // โดยดูจาก URL หรือ request parameter
            $productId = $request->input('product_id');
            if ($productId) {
                $product = Product::find($productId);
                if ($product) {
                    // ตรวจสอบว่ามี assessment draft อยู่แล้วหรือไม่
                    $assessment = Assessment::where('product_id', $product->id)
                        ->where('status', 'draft')
                        ->first();
                }
            }
        }

        if (!$product) {
            return redirect()->route('products.create')
                ->with('error', 'กรุณากรอกข้อมูลผลิตภัณฑ์ก่อน');
        }

        try {
            if (!$assessment) {
                // สร้าง assessment ใหม่
                $assessment = Assessment::create([
                    'product_id' => $product->id,
                    'question_group_id' => QuestionGroup::active()->first()->id,
                    'user_id' => Auth::user()->Customer_ID,
                    'CusIns_ID' => Auth::user()->Customer_ID,
                    'CusUdp_ID' => Auth::user()->Customer_ID,
                    'status' => 'draft',
                    'started_at' => now(),
                    'submitted_at' => null,
                    'DtmUpd' => now(), // ตั้งค่าวันที่อัปเดตครั้งแรก
                ]);
                
                Log::info('New assessment draft created:', ['assessment_id' => $assessment->id]);
            } else {
                // อัปเดต assessment ที่มีอยู่
                $assessment->CusUdp_ID = Auth::user()->Customer_ID;
                $assessment->DtmUpd = now(); // อัปเดตวันที่อัปเดตล่าสุด
                $assessment->save(); // บันทึกการเปลี่ยนแปลง
                
                Log::info('Existing assessment draft updated:', [
                    'assessment_id' => $assessment->id,
                    'new_DtmUpd' => $assessment->DtmUpd
                ]);
            }

            // ปิด foreign key checks ชั่วคราว
            DB::statement('SET FOREIGN_KEY_CHECKS = 0');

            // ลบคำตอบเก่า (ถ้ามี)
            $assessment->answers()->delete();

            // บันทึกคำตอบใหม่ (ถ้ามี)
            if ($request->has('answers') && is_array($request->answers)) {
                foreach ($request->answers as $questionId => $answerData) {
                    Log::info('Processing draft answer:', [
                        'question_id' => $questionId,
                        'answer_data' => $answerData
                    ]);

                    $questionId = (int) $questionId;
                    $question = \App\Models\Question::find($questionId);
                    
                    if (!$question) {
                        Log::error('Question not found for draft:', ['question_id' => $questionId]);
                        continue;
                    }

                    $answerText = null;
                    $comment = null;

                    if (isset($answerData['choice'])) {
                        if ($answerData['choice'] === 'yes') {
                            $answerText = 'ใช่';
                        } elseif ($answerData['choice'] === 'no') {
                            $answerText = 'ไม่ใช่';
                        } else {
                            $answerText = $answerData['choice'];
                        }
                    } elseif (isset($answerData['text'])) {
                        $answerText = $answerData['text'];
                    }

                    if (isset($answerData['details'])) {
                        $comment = $answerData['details'];
                    }

                    // บันทึกคำตอบสำหรับคำถามที่มี question_type ไม่เป็น null
                    if ($answerText && $question->question_type !== null) {
                        try {
                            $answer = Answer::create([
                                'assessment_id' => $assessment->id,
                                'question_id' => $questionId,
                                'answer_text' => $answerText,
                                'image_path' => null,
                                'comment' => $comment,
                                'CusIns_ID' => Auth::user()->Customer_ID,
                                'CusUdp_ID' => Auth::user()->Customer_ID,
                            ]);
                            
                            Log::info('Draft answer created:', [
                                'answer_id' => $answer->id,
                                'question_id' => $questionId,
                                'answer_text' => $answerText
                            ]);
                        } catch (\Exception $e) {
                            Log::error('Error creating draft answer:', [
                                'question_id' => $questionId,
                                'error' => $e->getMessage()
                            ]);
                            throw $e;
                        }
                    }
                }
            }

            // ประมวลผล parent questions ที่มี question_type = null และ use_in_evaluation = 1
            $parentQuestions = \App\Models\Question::where('question_type', null)
                ->where('use_in_evaluation', 1)
                ->whereHas('questionGroup', function($query) {
                    $query->where('is_active', true);
                })
                ->get();

            // ฟังก์ชันสำหรับคำนวณผลลัพธ์ของ parent question
            $calculateParentResult = function($parentQuestionId) use ($assessment, &$calculateParentResult) {
                $childQuestions = \App\Models\Question::where('parent_id', $parentQuestionId)
                    ->where('use_in_evaluation', 1)
                    ->get();

                if ($childQuestions->count() == 0) {
                    return false;
                }

                $childAnswers = [];
                foreach ($childQuestions as $childQuestion) {
                    if ($childQuestion->question_type === 'yes_no') {
                        // สำหรับ yes_no questions
                        $childAnswer = $assessment->answers()->where('question_id', $childQuestion->id)->first();
                        if ($childAnswer) {
                            $childAnswers[] = $childAnswer->answer_text === 'ใช่';
                        } else {
                            $childAnswers[] = false;
                        }
                    } elseif ($childQuestion->question_type === null) {
                        // สำหรับ nested parent questions - เรียก recursive
                        $childAnswers[] = $calculateParentResult($childQuestion->id);
                    }
                }

                // คำนวณผลลัพธ์ตาม logic operator
                $parentQuestion = \App\Models\Question::find($parentQuestionId);
                if ($parentQuestion->logic_operator === 'AND') {
                    return !in_array(false, $childAnswers);
                } elseif ($parentQuestion->logic_operator === 'OR') {
                    return in_array(true, $childAnswers);
                }
                
                return false;
            };

            foreach ($parentQuestions as $parentQuestion) {
                $result = $calculateParentResult($parentQuestion->id);

                // บันทึกผลลัพธ์ของ parent question
                try {
                    $parentAnswer = Answer::create([
                        'assessment_id' => $assessment->id,
                        'question_id' => $parentQuestion->id,
                        'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                        'image_path' => null,
                        'comment' => null,
                        'CusIns_ID' => Auth::user()->Customer_ID,
                        'CusUdp_ID' => Auth::user()->Customer_ID,
                    ]);
                    
                    Log::info('Parent draft answer created:', [
                        'answer_id' => $parentAnswer->id,
                        'question_id' => $parentQuestion->id,
                        'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                        'logic_operator' => $parentQuestion->logic_operator
                    ]);
                } catch (\Exception $e) {
                    Log::error('Error creating parent draft answer:', [
                        'question_id' => $parentQuestion->id,
                        'error' => $e->getMessage()
                    ]);
                    throw $e;
                }
            }

            // เปิด foreign key checks กลับ
            DB::statement('SET FOREIGN_KEY_CHECKS = 1');

            return redirect()->route('products.index')
                ->with('success', 'บันทึกฉบับร่างเรียบร้อยแล้ว คุณสามารถกดปุ่ม "ทำต่อ" เพื่อทำแบบประเมินต่อได้');

        } catch (\Exception $e) {
            // เปิด foreign key checks กลับในกรณี error
            DB::statement('SET FOREIGN_KEY_CHECKS = 1');
            
            Log::error('Assessment draft store error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'เกิดข้อผิดพลาดในการบันทึกฉบับร่าง: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * แสดงฟอร์มทำต่อจากฉบับร่าง
     */
    public function continueDraft(Product $product)
    {
        // ตรวจสอบสิทธิ์ - อนุญาตให้เจ้าของผลิตภัณฑ์, คนในบริษัทเดียวกัน, หรือ admin เข้าถึงได้
        $user = Auth::user();
        $hasAccess = false;
        
        // เจ้าของผลิตภัณฑ์
        if($user->Customer_ID === $product->user_id) {
            $hasAccess = true;
        }
        // Admin หรือ SuperAdmin
        elseif(strtolower($user->Level) === 'admin' || strtolower($user->Level) === 'superadmin') {
            $hasAccess = true;
        }
        // คนในบริษัทเดียวกัน
        elseif($user->Agency_ID && $product->user && $product->user->Agency_ID && $user->Agency_ID === $product->user->Agency_ID) {
            $hasAccess = true;
        }
        
        if(!$hasAccess) {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        // ตรวจสอบว่ามี assessment draft อยู่หรือไม่
        $assessment = Assessment::where('product_id', $product->id)
            ->where('status', 'draft')
            ->first();

        if (!$assessment) {
            return redirect()->route('products.index')
                ->with('error', 'ไม่พบฉบับร่างสำหรับผลิตภัณฑ์นี้');
        }

        // ดึงข้อมูลคำถามและ load answers พร้อม relationship
        $questionGroups = QuestionGroup::active()->with(['questions'])->get();
        $assessment->load(['answers' => function($query) {
            $query->orderBy('question_id');
        }]);
        
        // สร้าง validated array สำหรับแสดงข้อมูลผลิตภัณฑ์
        $validated = [
            'name' => $product->name,
            'model' => $product->model,
            'product_type' => $product->product_type,
            'log_receive_method' => $product->log_receive_method,
            'description' => $product->description
        ];
        
        // สร้าง array ของคำตอบเพื่อใช้ใน view
        $existingAnswers = [];
        foreach ($assessment->answers as $answer) {
            $existingAnswers[$answer->question_id] = [
                'choice' => $answer->answer_text,
                'details' => $answer->comment,
                'text' => $answer->answer_text
            ];
        }
        
        return view('assessments.manual', compact('assessment', 'questionGroups', 'validated', 'existingAnswers'));
    }

    /**
     * แสดงฟอร์มแก้ไขการประเมิน
     */
    public function edit(Assessment $assessment)
    {

        // ตรวจสอบสิทธิ์
        if (Auth::user()->Customer_ID !== $assessment->user_id && strtolower(Auth::user()->Level) !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }
        Log::info('User level: ' . Auth::user()->Level);
        Log::info('User Customer_ID: ' . Auth::user()->Customer_ID);
        Log::info('Assessment user_id: ' . $assessment->user_id);

        // ดึงข้อมูลคำถาม
        $questionGroups = QuestionGroup::active()->with(['questions'])->get();
        
        return view('assessments.edit', compact('assessment', 'questionGroups'));
    }

    /**
     * แสดงหน้าแบบประเมินคู่มือ
     */
    public function manual(Assessment $assessment)
    {
        // ตรวจสอบสิทธิ์
        if (Auth::user()->Customer_ID !== $assessment->user_id && strtolower(Auth::user()->Level) !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        // ตรวจสอบว่าเป็น draft หรือไม่
        if ($assessment->status !== 'draft') {
            return redirect()->route('assessments.edit', $assessment)
                ->with('error', 'ไม่สามารถเข้าถึงหน้าแบบประเมินคู่มือได้');
        }

        // ดึงข้อมูลคำถามและ load answers
        $questionGroups = QuestionGroup::active()->with(['questions'])->get();
        $assessment->load('answers');
        
        return view('assessments.manual', compact('assessment', 'questionGroups'));
    }

    /**
     * อัปเดตการประเมิน
     */
    public function update(Request $request, Assessment $assessment)
    {
        // ตรวจสอบสิทธิ์
        if (Auth::user()->Customer_ID !== $assessment->user_id && strtolower(Auth::user()->Level) !== 'admin') {
            abort(403, 'ไม่มีสิทธิ์เข้าถึงข้อมูลนี้');
        }

        $request->validate([
            'answers' => 'nullable|array',
            'answers.*.choice' => 'nullable|string',
            'answers.*.details' => 'nullable|string',
            'answers.*.text' => 'nullable|string',
        ]);

        try {
            // อัปเดตสถานะเป็น submitted
            $assessment->update([
                'status' => 'submitted',
                'submitted_at' => now(),
                'CusUdp_ID' => Auth::user()->Customer_ID,
            ]);

            // ปิด foreign key checks ชั่วคราว
            DB::statement('SET FOREIGN_KEY_CHECKS = 0');

            // ลบคำตอบเก่า
            $assessment->answers()->delete();

            // บันทึกคำตอบใหม่
            if ($request->has('answers') && is_array($request->answers)) {
                foreach ($request->answers as $questionId => $answerData) {
                    $questionId = (int) $questionId;
                    $question = \App\Models\Question::find($questionId);
                    
                    if (!$question) {
                        continue;
                    }

                    $answerText = null;
                    $comment = null;

                    if (isset($answerData['choice'])) {
                        if ($answerData['choice'] === 'yes') {
                            $answerText = 'ใช่';
                        } elseif ($answerData['choice'] === 'no') {
                            $answerText = 'ไม่ใช่';
                        } else {
                            $answerText = $answerData['choice'];
                        }
                    } elseif (isset($answerData['text'])) {
                        $answerText = $answerData['text'];
                    }

                    if (isset($answerData['details'])) {
                        $comment = $answerData['details'];
                    }

                    // บันทึกคำตอบสำหรับคำถามที่มี question_type ไม่เป็น null
                    if ($answerText && $question->question_type !== null) {
                        Answer::create([
                            'assessment_id' => $assessment->id,
                            'question_id' => $questionId,
                            'answer_text' => $answerText,
                            'image_path' => null,
                            'comment' => $comment,
                            'CusIns_ID' => Auth::user()->Customer_ID,
                            'CusUdp_ID' => Auth::user()->Customer_ID,
                        ]);
                    }
                }
            }

            // ประมวลผล parent questions ที่มี question_type = null และ use_in_evaluation = 1
            $parentQuestions = \App\Models\Question::where('question_type', null)
                ->where('use_in_evaluation', 1)
                ->whereHas('questionGroup', function($query) {
                    $query->where('is_active', true);
                })
                ->get();

            // ฟังก์ชันสำหรับคำนวณผลลัพธ์ของ parent question
            $calculateParentResult = function($parentQuestionId) use ($assessment, &$calculateParentResult) {
                $childQuestions = \App\Models\Question::where('parent_id', $parentQuestionId)
                    ->where('use_in_evaluation', 1)
                    ->get();

                if ($childQuestions->count() == 0) {
                    return false;
                }

                $childAnswers = [];
                foreach ($childQuestions as $childQuestion) {
                    if ($childQuestion->question_type === 'yes_no') {
                        // สำหรับ yes_no questions
                        $childAnswer = $assessment->answers()->where('question_id', $childQuestion->id)->first();
                        if ($childAnswer) {
                            $childAnswers[] = $childAnswer->answer_text === 'ใช่';
                        } else {
                            $childAnswers[] = false;
                        }
                    } elseif ($childQuestion->question_type === null) {
                        // สำหรับ nested parent questions - เรียก recursive
                        $childAnswers[] = $calculateParentResult($childQuestion->id);
                    }
                }

                // คำนวณผลลัพธ์ตาม logic operator
                $parentQuestion = \App\Models\Question::find($parentQuestionId);
                if ($parentQuestion->logic_operator === 'AND') {
                    return !in_array(false, $childAnswers);
                } elseif ($parentQuestion->logic_operator === 'OR') {
                    return in_array(true, $childAnswers);
                }
                
                return false;
            };

            foreach ($parentQuestions as $parentQuestion) {
                $result = $calculateParentResult($parentQuestion->id);

                // บันทึกผลลัพธ์ของ parent question
                Answer::create([
                    'assessment_id' => $assessment->id,
                    'question_id' => $parentQuestion->id,
                    'answer_text' => $result ? 'ใช่' : 'ไม่ใช่',
                    'image_path' => null,
                    'comment' => null,
                    'CusIns_ID' => Auth::user()->Customer_ID,
                    'CusUdp_ID' => Auth::user()->Customer_ID,
                ]);
            }

            // เปิด foreign key checks กลับ
            DB::statement('SET FOREIGN_KEY_CHECKS = 1');

            return redirect()->route('products.report', $assessment->product)
                ->with('success', 'ส่งแบบประเมินเรียบร้อยแล้ว');

        } catch (\Exception $e) {
            // เปิด foreign key checks กลับในกรณี error
            DB::statement('SET FOREIGN_KEY_CHECKS = 1');
            
            return redirect()->back()
                ->with('error', 'เกิดข้อผิดพลาดในการส่งแบบประเมิน: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * ดึงรายการบริษัทที่ consult user สามารถเข้าถึงได้
     */
    private function getConsultUserAgencies($user)
    {
        // ดึงจากตาราง agency_customer
        $agencyIds = \App\Models\AgencyCustomer::where('Customer_ID', $user->Customer_ID)
            ->pluck('Agency_ID')
            ->toArray();
            
        // เพิ่มบริษัทหลักจาก customer table ด้วย
        if ($user->Agency_ID && !in_array($user->Agency_ID, $agencyIds)) {
            $agencyIds[] = $user->Agency_ID;
        }
        
        return \App\Models\Agency::whereIn('Agency_ID', $agencyIds)->get();
    }
} 

