<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Question;
use App\Models\QuestionGroup;

class ShowQuestions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'questions:show';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'แสดงข้อมูลคำถามทั้งหมดในตาราง questions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== ข้อมูลคำถามทั้งหมดในตาราง questions ===');
        
        // ดึงข้อมูลคำถามทั้งหมดพร้อมกับข้อมูลที่เกี่ยวข้อง
        $questions = Question::with(['questionGroup', 'parent'])
            ->orderBy('question_group_id')
            ->orderBy('sort_order')
            ->get();

        if ($questions->count() === 0) {
            $this->warn('ไม่พบข้อมูลคำถามในตาราง');
            return;
        }

        // สร้างตารางข้อมูล
        $headers = [
            'ID', 'Group', 'Level', 'Parent ID', 'Question Text', 
            'Question Type', 'Use in Evaluation', 'Choices', 'Display Type'
        ];

        $rows = [];
        foreach ($questions as $question) {
            $level = $this->getQuestionLevel($question, $questions);
            $displayType = $this->getDisplayType($question->use_in_evaluation, $question->question_type);
            
            $rows[] = [
                $question->id,
                $question->questionGroup->name ?? 'ไม่มี',
                $level,
                $question->parent_id ?? 'ไม่มี',
                $this->truncateText($question->question_text, 50),
                $question->question_type ?? 'null',
                $question->use_in_evaluation ? 'ใช่' : 'ไม่ใช่',
                $question->choices ? 'มี' : 'ไม่มี',
                $displayType
            ];
        }

        $this->table($headers, $rows);

        // แสดงสถิติ
        $this->showStatistics($questions);
    }

    /**
     * ฟังก์ชันสำหรับกำหนดระดับของคำถาม
     */
    private function getQuestionLevel($question, $allQuestions)
    {
        if (!$question->parent_id) {
            return "Parent (ระดับที่ 1)";
        }
        
        $level = 1;
        $currentParentId = $question->parent_id;
        
        while ($currentParentId) {
            $parent = $allQuestions->where('id', $currentParentId)->first();
            if ($parent) {
                $level++;
                $currentParentId = $parent->parent_id;
            } else {
                break;
            }
        }
        
        switch ($level) {
            case 1: return "Parent (ระดับที่ 1)";
            case 2: return "Child (ระดับที่ 2)";
            case 3: return "Grand Child (ระดับที่ 3)";
            case 4: return "Great Grand Child (ระดับที่ 4)";
            case 5: return "Great Great Grand Child (ระดับที่ 5)";
            case 6: return "Great Great Great Grand Child (ระดับที่ 6)";
            default: return "ระดับที่ {$level}";
        }
    }

    /**
     * ฟังก์ชันสำหรับกำหนดประเภทการแสดงผล
     */
    private function getDisplayType($useInEvaluation, $questionType)
    {
        if (!$useInEvaluation && $questionType === null) {
            return "Accordion Button";
        } elseif ($useInEvaluation && $questionType !== null) {
            return "Card ปกติ (ใช้ในการประเมิน)";
        } elseif (!$useInEvaluation && $questionType !== null) {
            return "Card สีเทา (ไม่ใช้ในการประเมิน)";
        } else {
            return "ไม่ระบุ";
        }
    }

    /**
     * ฟังก์ชันสำหรับตัดข้อความ
     */
    private function truncateText($text, $length)
    {
        if (strlen($text) <= $length) {
            return $text;
        }
        return substr($text, 0, $length) . '...';
    }

    /**
     * ฟังก์ชันสำหรับแสดงสถิติ
     */
    private function showStatistics($questions)
    {
        $this->newLine();
        $this->info('=== สรุปสถิติ ===');
        
        $totalQuestions = $questions->count();
        $parentQuestions = $questions->where('parent_id', null)->count();
        $childQuestions = $questions->where('parent_id', '!=', null)->count();
        
        $accordionQuestions = $questions->where('use_in_evaluation', false)->where('question_type', null)->count();
        $evaluationQuestions = $questions->where('use_in_evaluation', true)->where('question_type', '!=', null)->count();
        $nonEvaluationQuestions = $questions->where('use_in_evaluation', false)->where('question_type', '!=', null)->count();
        
        $this->line("จำนวนคำถามทั้งหมด: {$totalQuestions}");
        $this->line("Parent Questions: {$parentQuestions}");
        $this->line("Child Questions: {$childQuestions}");
        $this->line("Accordion Questions: {$accordionQuestions}");
        $this->line("Evaluation Questions: {$evaluationQuestions}");
        $this->line("Non-Evaluation Questions: {$nonEvaluationQuestions}");
        
        // แสดงข้อมูลตาม Group
        $this->newLine();
        $this->info('=== ข้อมูลตาม Group ===');
        
        $groupStats = $questions->groupBy('question_group_id');
        foreach ($groupStats as $groupId => $groupQuestions) {
            $groupName = $groupQuestions->first()->questionGroup->name ?? 'ไม่มีชื่อ';
            $this->line("Group: {$groupName} (ID: {$groupId}) - จำนวนคำถาม: {$groupQuestions->count()}");
        }
    }
} 